package codegen

import "goa.design/goa/v3/eval"

type (
	// GenerateFunc makes it possible to modify the files generated by the
	// goa code generators and other plugins. A GenerateFunc accepts the Go
	// import path of the "gen" package, the design roots as well as the
	// currently generated files (produced initially by the goa generators
	// and potentially modified by previously run plugins) and returns a new
	// set of files.
	GenerateFunc func(genpkg string, roots []eval.Root, files []*File) ([]*File, error)

	// PrepareFunc makes it possible to modify the design roots before
	// the files being generated by the goa code generators or other plugins.
	PrepareFunc func(genpkg string, roots []eval.Root) error

	// plugin is a plugin that has been registered with a given command.
	plugin struct {
		// PrepareFunc is the plugin preparation function.
		PrepareFunc
		// GenerateFunc is the plugin generator function.
		GenerateFunc
		// name is the plugin name.
		name string
		// cmd is the name of cmd to run.
		cmd string
		// if first is set the plugin cmd must run before all other plugins.
		first bool
		// if last is set the plugin cmd must run after all other plugins.
		last bool
	}
)

// plugins keeps track of the registered plugins sorted by their first/last bools,
// names, or registration order.
var plugins []*plugin

// RegisterPlugin adds the plugin to the list of plugins to be invoked with the
// given command.
func RegisterPlugin(name string, cmd string, pre PrepareFunc, p GenerateFunc) {
	np := &plugin{name: name, PrepareFunc: pre, GenerateFunc: p, cmd: cmd}
	var inserted bool
	for i, plgn := range plugins {
		if plgn.last || (!plgn.first && np.name < plgn.name) {
			plugins = append(plugins[:i], append([]*plugin{np}, plugins[i:]...)...)
			inserted = true
			break
		}
	}
	if !inserted {
		plugins = append(plugins, np)
	}
}

// RegisterPluginFirst adds the plugin to the beginning of the list of plugins
// to be invoked with the given command. If more than one plugins are registered
// using this, the plugins will be sorted alphabetically by their names. If two
// plugins have same names, then they are sorted by registration order.
func RegisterPluginFirst(name string, cmd string, pre PrepareFunc, p GenerateFunc) {
	np := &plugin{name: name, PrepareFunc: pre, GenerateFunc: p, cmd: cmd, first: true}
	var inserted bool
	for i, plgn := range plugins {
		if !plgn.first || np.name < plgn.name {
			plugins = append(plugins[:i], append([]*plugin{np}, plugins[i:]...)...)
			inserted = true
			break
		}
	}
	if !inserted {
		plugins = append(plugins, np)
	}
}

// RegisterPluginLast adds the plugin to the end of the list of plugins
// to be invoked with the given command. If more than one plugins are registered
// using this, the plugins will be sorted alphabetically by their names. If two
// plugins have same names, then they are sorted by registration order.
func RegisterPluginLast(name string, cmd string, pre PrepareFunc, p GenerateFunc) {
	np := &plugin{name: name, PrepareFunc: pre, GenerateFunc: p, cmd: cmd, last: true}
	var inserted bool
	for i := len(plugins) - 1; i >= 0; i-- {
		plgn := plugins[i]
		if !plgn.last || plgn.name < np.name {
			plugins = append(plugins[:i+1], append([]*plugin{np}, plugins[i+1:]...)...)
			inserted = true
			break
		}
	}
	if !inserted {
		plugins = append(plugins, np)
	}
}

// RunPluginsPrepare executes the plugins prepare functions in the order
// they were registered.
func RunPluginsPrepare(cmd, genpkg string, roots []eval.Root) error {
	for _, plugin := range plugins {
		if plugin.cmd != cmd {
			continue
		}
		if plugin.PrepareFunc != nil {
			err := plugin.PrepareFunc(genpkg, roots)
			if err != nil {
				return err
			}
		}
	}
	return nil
}

// RunPlugins executes the plugins registered with the given command in the order
// they were registered.
func RunPlugins(cmd, genpkg string, roots []eval.Root, genfiles []*File) ([]*File, error) {
	for _, plugin := range plugins {
		if plugin.cmd != cmd {
			continue
		}
		gs, err := plugin.GenerateFunc(genpkg, roots, genfiles)
		if err != nil {
			return nil, err
		}
		genfiles = gs
	}
	return genfiles, nil
}
