package io

import (
	"errors"
	"fmt"
	"io"
	"os"
	"path/filepath"
	"sort"
	"strings"
	"text/template"

	"git.sr.ht/~charles/rq/internal"
	"git.sr.ht/~charles/rq/util"
)

var templateFuncs template.FuncMap

var outputHandlers map[string]func() OutputHandler

// OutputHandler represents a handler for one type of output format, such as JSON
// or CSV.
type OutputHandler interface {

	// Format marshals the given data to the relevant format and then
	// writes it to the specified writer.
	Format(writer io.Writer, data interface{}) error

	// SetOption specifies an option to control how the handler should
	// behave. Setting options that the handler does not implement should
	// cause SetOption to return nil, this way the CLI handling logic does
	// not have to determine which handler is being used before deciding
	// which options to try to set.
	//
	// The following options should be supported by all output formatters
	// to the greatest extent possible:
	//
	// output.colorize (bool) if true, syntax highlight the output if
	// possible, otherwise do not do so (defaults to true)
	//
	// output.pretty (bool) if true, pretty-print the output if possible,
	// otherwise minify the output or otherwise leave in the default format
	// (defaults to true)
	//
	// output.indent (string) for formats such as JSON and YAML where
	// indenting is used, specify the indent character (defaults to '\t')
	//
	// output.style (string) specifies the Chroma format to use when
	// colorizing output (default: "native"); see:
	// https://xyproto.github.io/splash/docs/
	SetOption(name string, value string) error

	// Name should return the name of this handler, e.g. "json" or "csv".
	Name() string
}

// SelectOutputHandler chooses a handler based on the name provided.
//
// If name is the empty string, then it default to "json".
func SelectOutputHandler(name string) (OutputHandler, error) {
	if name == "" {
		name = "json"
	}

	h, ok := outputHandlers[name]
	if !ok {
		return nil, fmt.Errorf("no such handler with name '%s'", name)
	}

	return h(), nil
}

// ListOutputHandlers lists all known handler names.
func ListOutputHandlers() []string {
	l := []string{}
	for h := range outputHandlers {
		l = append(l, h)
	}
	sort.Strings(l)
	return l
}

func registerOutputHandler(name string, newFunc func() OutputHandler) {
	if outputHandlers == nil {
		outputHandlers = make(map[string]func() OutputHandler)
	}

	outputHandlers[name] = newFunc
}

// WriteOutput uses the given DataSpec to select an appropriate output handler
// and write the given data out using it to the file it specifies.
//
// If the DataSpec's FilePath is empty, then the options
// generic.empty_file_path is set to true.
func WriteOutput(data interface{}, d *DataSpec) error {
	if d.FilePath == "" {
		return errors.New("Cannot write data to empty FilePath")
	}

	f, err := os.OpenFile(d.FilePath, os.O_TRUNC|os.O_WRONLY|os.O_CREATE, 0644)
	if err != nil {
		return err
	}
	defer func() { _ = f.Close() }()

	return WriteOutputToWriter(data, d, f)
}

// WriteOutputToWriter works similarly to WriteOutput, but ignores the
// DataSpec's FilePath, instead writing all data to the given writer.
//
// If the DataSpec's FilePath is empty, then the options
// generic.empty_file_path is set to true.
func WriteOutputToWriter(data interface{}, d *DataSpec, w io.Writer) error {

	// Resolve the format to the given format, the extension, or JSON.
	format := d.Format
	if format == "" {
		format = strings.ToLower(filepath.Ext(d.FilePath))
	}
	handler, err := SelectOutputHandler(format)
	if err != nil {
		handler, err = SelectOutputHandler("json")
		if err != nil {
			// we should always be able to select the JSON handler
			panic(err)
		}
	}

	if d.Options != nil {
		for o, v := range d.Options {
			if err := handler.SetOption(o, v); err != nil {
				return err
			}
		}
	}

	if d.FilePath == "" {
		err := handler.SetOption("generic.empty_file_path", "true")
		if err != nil {
			return err
		}
	}

	return handler.Format(w, data)
}

func init() {
	templateFuncs = util.GetTemplateFuncs()
	templateFuncs["sfake"] = internal.SfakeTemplate
}
