// Copyright (c) 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/strings/stringprintf.h"
#include "components/autofill/content/renderer/form_autofill_util.h"

#include "base/cxx17_backports.h"
#include "base/metrics/field_trial.h"
#include "base/strings/strcat.h"
#include "base/strings/string_number_conversions.h"
#include "base/strings/utf_string_conversions.h"
#include "base/test/scoped_feature_list.h"
#include "components/autofill/core/common/autofill_features.h"
#include "components/autofill/core/common/mojom/autofill_types.mojom-shared.h"
#include "components/autofill/core/common/unique_ids.h"
#include "content/public/renderer/render_view.h"
#include "content/public/test/render_view_test.h"
#include "content/public/test/test_utils.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/blink/public/platform/web_string.h"
#include "third_party/blink/public/platform/web_vector.h"
#include "third_party/blink/public/web/web_document.h"
#include "third_party/blink/public/web/web_element.h"
#include "third_party/blink/public/web/web_element_collection.h"
#include "third_party/blink/public/web/web_form_control_element.h"
#include "third_party/blink/public/web/web_form_element.h"
#include "third_party/blink/public/web/web_frame_widget.h"
#include "third_party/blink/public/web/web_input_element.h"
#include "third_party/blink/public/web/web_local_frame.h"
#include "third_party/blink/public/web/web_remote_frame.h"
#include "third_party/blink/public/web/web_select_element.h"
#include "third_party/blink/public/web/web_view.h"

using autofill::mojom::ButtonTitleType;
using blink::WebDocument;
using blink::WebElement;
using blink::WebElementCollection;
using blink::WebFormControlElement;
using blink::WebFormElement;
using blink::WebInputElement;
using blink::WebLocalFrame;
using blink::WebNode;
using blink::WebSelectElement;
using blink::WebString;
using blink::WebVector;

namespace autofill {
namespace form_util {
namespace {

struct AutofillFieldLabelSourceCase {
  const char* html;
  const FormFieldData::LabelSource label_source;
};

struct AutofillFieldUtilCase {
  const char* description;
  const char* html;
  const char16_t* expected_label;
};

const char kElevenChildren[] =
    "<div id='target'>"
    "<div>child0</div>"
    "<div>child1</div>"
    "<div>child2</div>"
    "<div>child3</div>"
    "<div>child4</div>"
    "<div>child5</div>"
    "<div>child6</div>"
    "<div>child7</div>"
    "<div>child8</div>"
    "<div>child9</div>"
    "<div>child10</div>"
    "</div>";
const char16_t kElevenChildrenExpected[] =
    u"child0child1child2child3child4child5child6child7child8";

const char kElevenChildrenNested[] =
    "<div id='target'>"
    "<div>child0"
    "<div>child1"
    "<div>child2"
    "<div>child3"
    "<div>child4"
    "<div>child5"
    "<div>child6"
    "<div>child7"
    "<div>child8"
    "<div>child9"
    "<div>child10"
    "</div></div></div></div></div></div></div></div></div></div></div></div>";
// Take 10 elements -1 for target element, -1 as text is a leaf element.
const char16_t kElevenChildrenNestedExpected[] =
    u"child0child1child2child3child4";

const char kSkipElement[] =
    "<div id='target'>"
    "<div>child0</div>"
    "<div class='skip'>child1</div>"
    "<div>child2</div>"
    "</div>";
// TODO(crbug.com/796918): Should be child0child2
const char16_t kSkipElementExpected[] = u"child0";

const char kDivTableExample1[] =
    "<div>"
    "<div>label</div><div><input id='target'/></div>"
    "</div>";
const char16_t kDivTableExample1Expected[] = u"label";

const char kDivTableExample2[] =
    "<div>"
    "<div>label</div>"
    "<div>should be skipped<input/></div>"
    "<div><input id='target'/></div>"
    "</div>";
const char16_t kDivTableExample2Expected[] = u"label";

const char kDivTableExample3[] =
    "<div>"
    "<div>should be skipped<input/></div>"
    "<div>label</div>"
    "<div><input id='target'/></div>"
    "</div>";
const char16_t kDivTableExample3Expected[] = u"label";

const char kDivTableExample4[] =
    "<div>"
    "<div>should be skipped<input/></div>"
    "label"
    "<div><input id='target'/></div>"
    "</div>";
// TODO(crbug.com/796918): Should be label
const char16_t kDivTableExample4Expected[] = u"";

const char kDivTableExample5[] =
    "<div>"
    "<div>label<div><input id='target'/></div>behind</div>"
    "</div>";
// TODO(crbug.com/796918): Should be label
const char16_t kDivTableExample5Expected[] = u"labelbehind";

const char kDivTableExample6[] =
    "<div>"
    "<div>label<div><div>-<div><input id='target'/></div></div>"
    "</div>";
// TODO(crbug.com/796918): Should be "label" or "label-"
const char16_t kDivTableExample6Expected[] = u"";

void VerifyButtonTitleCache(const WebFormElement& form_target,
                            const ButtonTitleList& expected_button_titles,
                            const ButtonTitlesCache& actual_cache) {
  EXPECT_THAT(actual_cache,
              testing::ElementsAre(testing::Pair(GetFormRendererId(form_target),
                                                 expected_button_titles)));
}

FrameToken GetFrameToken(const blink::WebElement& iframe_element) {
  blink::WebFrame* frame =
      blink::WebFrame::FromFrameOwnerElement(iframe_element);
  if (frame && frame->IsWebLocalFrame()) {
    return LocalFrameToken(
        frame->ToWebLocalFrame()->GetLocalFrameToken().value());
  } else if (frame && frame->IsWebRemoteFrame()) {
    return RemoteFrameToken(
        frame->ToWebRemoteFrame()->GetRemoteFrameToken().value());
  } else {
    NOTREACHED();
    return FrameToken();
  }
}

class FormAutofillUtilsTest : public content::RenderViewTest {
 public:
  FormAutofillUtilsTest() {}
  ~FormAutofillUtilsTest() override {}
};

TEST_F(FormAutofillUtilsTest, FindChildTextTest) {
  static const AutofillFieldUtilCase test_cases[] = {
      {"simple test", "<div id='target'>test</div>", u"test"},
      {"Concatenate test", "<div id='target'><span>one</span>two</div>",
       u"onetwo"},
      // TODO(crbug.com/796918): should be "onetwo"
      {"Ignore input", "<div id='target'>one<input value='test'/>two</div>",
       u"one"},
      {"Trim", "<div id='target'>   one<span>two  </span></div>", u"onetwo"},
      {"eleven children", kElevenChildren, kElevenChildrenExpected},
      // TODO(crbug.com/796918): Depth is only 5 elements
      {"eleven children nested", kElevenChildrenNested,
       kElevenChildrenNestedExpected},
  };
  for (auto test_case : test_cases) {
    SCOPED_TRACE(test_case.description);
    LoadHTML(test_case.html);
    WebLocalFrame* web_frame = GetMainFrame();
    ASSERT_NE(nullptr, web_frame);
    WebElement target = web_frame->GetDocument().GetElementById("target");
    ASSERT_FALSE(target.IsNull());
    EXPECT_EQ(test_case.expected_label, FindChildText(target));
  }
}

TEST_F(FormAutofillUtilsTest, FindChildTextSkipElementTest) {
  static const AutofillFieldUtilCase test_cases[] = {
      {"Skip div element", kSkipElement, kSkipElementExpected},
  };
  for (auto test_case : test_cases) {
    SCOPED_TRACE(test_case.description);
    LoadHTML(test_case.html);
    WebLocalFrame* web_frame = GetMainFrame();
    ASSERT_NE(nullptr, web_frame);
    WebElement target = web_frame->GetDocument().GetElementById("target");
    ASSERT_FALSE(target.IsNull());
    WebVector<WebElement> web_to_skip =
        web_frame->GetDocument().QuerySelectorAll("div[class='skip']");
    std::set<WebNode> to_skip;
    for (size_t i = 0; i < web_to_skip.size(); ++i) {
      to_skip.insert(web_to_skip[i]);
    }

    EXPECT_EQ(test_case.expected_label,
              FindChildTextWithIgnoreListForTesting(target, to_skip));
  }
}

TEST_F(FormAutofillUtilsTest, InferLabelForElementTest) {
  static const AutofillFieldUtilCase test_cases[] = {
      {"DIV table test 1", kDivTableExample1, kDivTableExample1Expected},
      {"DIV table test 2", kDivTableExample2, kDivTableExample2Expected},
      {"DIV table test 3", kDivTableExample3, kDivTableExample3Expected},
      {"DIV table test 4", kDivTableExample4, kDivTableExample4Expected},
      {"DIV table test 5", kDivTableExample5, kDivTableExample5Expected},
      {"DIV table test 6", kDivTableExample6, kDivTableExample6Expected},
  };
  for (auto test_case : test_cases) {
    SCOPED_TRACE(test_case.description);
    LoadHTML(test_case.html);
    WebLocalFrame* web_frame = GetMainFrame();
    ASSERT_NE(nullptr, web_frame);
    WebElement target = web_frame->GetDocument().GetElementById("target");
    ASSERT_FALSE(target.IsNull());
    const WebFormControlElement form_target =
        target.ToConst<WebFormControlElement>();
    ASSERT_FALSE(form_target.IsNull());

    FormFieldData::LabelSource label_source =
        FormFieldData::LabelSource::kUnknown;
    std::u16string label;
    InferLabelForElementForTesting(form_target, &label, &label_source);
    EXPECT_EQ(test_case.expected_label, label);
  }
}

TEST_F(FormAutofillUtilsTest, InferLabelSourceTest) {
  const char16_t kLabelSourceExpectedLabel[] = u"label";
  static const AutofillFieldLabelSourceCase test_cases[] = {
      {"<div><div>label</div><div><input id='target'/></div></div>",
       FormFieldData::LabelSource::kDivTable},
      {"<label>label</label><input id='target'/>",
       FormFieldData::LabelSource::kLabelTag},
      {"<b>l</b><strong>a</strong>bel<input id='target'/>",
       FormFieldData::LabelSource::kCombined},
      {"<p><b>l</b><strong>a</strong>bel</p><input id='target'/>",
       FormFieldData::LabelSource::kPTag},
      {"<input id='target' placeholder='label'/>",
       FormFieldData::LabelSource::kPlaceHolder},
      {"<input id='target' aria-label='label'/>",
       FormFieldData::LabelSource::kAriaLabel},
      {"<input id='target' value='label'/>",
       FormFieldData::LabelSource::kValue},
      {"<li>label<div><input id='target'/></div></li>",
       FormFieldData::LabelSource::kLiTag},
      {"<table><tr><td>label</td><td><input id='target'/></td></tr></table>",
       FormFieldData::LabelSource::kTdTag},
      {"<dl><dt>label</dt><dd><input id='target'></dd></dl>",
       FormFieldData::LabelSource::kDdTag},
  };

  for (auto test_case : test_cases) {
    SCOPED_TRACE(testing::Message() << test_case.label_source);
    LoadHTML(test_case.html);
    WebLocalFrame* web_frame = GetMainFrame();
    ASSERT_NE(nullptr, web_frame);
    WebElement target = web_frame->GetDocument().GetElementById("target");
    ASSERT_FALSE(target.IsNull());
    const WebFormControlElement form_target =
        target.ToConst<WebFormControlElement>();
    ASSERT_FALSE(form_target.IsNull());

    FormFieldData::LabelSource label_source =
        FormFieldData::LabelSource::kUnknown;
    std::u16string label;
    EXPECT_TRUE(autofill::form_util::InferLabelForElementForTesting(
        form_target, &label, &label_source));
    EXPECT_EQ(kLabelSourceExpectedLabel, label);
    EXPECT_EQ(test_case.label_source, label_source);
  }
}

TEST_F(FormAutofillUtilsTest, GetButtonTitles) {
  constexpr char kHtml[] =
      "<form id='target'>"
      "  <input type='button' value='Clear field'>"
      "  <input type='button' value='Clear field'>"
      "  <input type='button' value='Clear field'>"
      "  <input type='button' value='\n Show\t password '>"
      "  <button>Sign Up</button>"
      "  <button type='button'>Register</button>"
      "  <a id='Submit' value='Create account'>"
      "  <div name='BTN'> Join </div>"
      "  <span class='button'> Start </span>"
      "  <a class='empty button' value='   \t   \n'>"
      "</form>";

  LoadHTML(kHtml);
  WebLocalFrame* web_frame = GetMainFrame();
  ASSERT_NE(nullptr, web_frame);
  const WebElement& target = web_frame->GetDocument().GetElementById("target");
  ASSERT_FALSE(target.IsNull());
  const WebFormElement& form_target = target.ToConst<WebFormElement>();
  ASSERT_FALSE(form_target.IsNull());
  ButtonTitlesCache cache;

  autofill::ButtonTitleList actual =
      GetButtonTitles(form_target, web_frame->GetDocument(), &cache);

  autofill::ButtonTitleList expected = {
      {u"Clear field", ButtonTitleType::INPUT_ELEMENT_BUTTON_TYPE},
      {u"Show password", ButtonTitleType::INPUT_ELEMENT_BUTTON_TYPE},
      {u"Sign Up", ButtonTitleType::BUTTON_ELEMENT_SUBMIT_TYPE},
      {u"Register", ButtonTitleType::BUTTON_ELEMENT_BUTTON_TYPE},
      {u"Create account", ButtonTitleType::HYPERLINK},
      {u"Join", ButtonTitleType::DIV},
      {u"Start", ButtonTitleType::SPAN}};
  EXPECT_EQ(expected, actual);

  VerifyButtonTitleCache(form_target, expected, cache);
}

TEST_F(FormAutofillUtilsTest, GetButtonTitles_TooLongTitle) {
  std::string title;
  for (int i = 0; i < 300; ++i)
    title += "a";
  std::string kFormHtml = "<form id='target'>";
  for (int i = 0; i < 10; i++) {
    std::string kFieldHtml =
        "<input type='button' value='" + base::NumberToString(i) + title + "'>";
    kFormHtml += kFieldHtml;
  }
  kFormHtml += "</form>";

  LoadHTML(kFormHtml.c_str());
  WebLocalFrame* web_frame = GetMainFrame();
  ASSERT_NE(nullptr, web_frame);
  const WebElement& target = web_frame->GetDocument().GetElementById("target");
  ASSERT_FALSE(target.IsNull());
  const WebFormElement& form_target = target.ToConst<WebFormElement>();
  ASSERT_FALSE(form_target.IsNull());
  ButtonTitlesCache cache;

  autofill::ButtonTitleList actual =
      GetButtonTitles(form_target, web_frame->GetDocument(), &cache);

  int total_length = 0;
  for (auto title : actual) {
    EXPECT_GE(30u, title.first.length());
    total_length += title.first.length();
  }
  EXPECT_EQ(200, total_length);
}

TEST_F(FormAutofillUtilsTest, GetButtonTitles_Formless) {
  constexpr char kNoFormHtml[] =
      "<div class='reg-form'>"
      "  <input type='button' value='\n Show\t password '>"
      "  <button>Sign Up</button>"
      "  <button type='button'>Register</button>"
      "</div>"
      "<form id='ignored-form'>"
      "  <input type='button' value='Ignore this'>"
      "  <button>Ignore this</button>"
      "  <a id='Submit' value='Ignore this'>"
      "  <div name='BTN'>Ignore this</div>"
      "</form>";

  LoadHTML(kNoFormHtml);
  WebLocalFrame* web_frame = GetMainFrame();
  ASSERT_NE(nullptr, web_frame);
  WebFormElement form_target;
  ASSERT_FALSE(web_frame->GetDocument().Body().IsNull());
  ButtonTitlesCache cache;

  autofill::ButtonTitleList actual =
      GetButtonTitles(form_target, web_frame->GetDocument(), &cache);
  autofill::ButtonTitleList expected = {
      {u"Show password", ButtonTitleType::INPUT_ELEMENT_BUTTON_TYPE},
      {u"Sign Up", ButtonTitleType::BUTTON_ELEMENT_SUBMIT_TYPE},
      {u"Register", ButtonTitleType::BUTTON_ELEMENT_BUTTON_TYPE}};
  EXPECT_EQ(expected, actual);

  VerifyButtonTitleCache(form_target, expected, cache);
}

TEST_F(FormAutofillUtilsTest, GetButtonTitles_DisabledIfNoCache) {
  // Button titles scraping for unowned forms can be time-consuming and disabled
  // in Beta and Stable. To disable button titles computation, |buttons_cache|
  // should be null.
  constexpr char kNoFormHtml[] =
      "<div class='reg-form'>"
      "  <input type='button' value='\n Show\t password '>"
      "  <button>Sign Up</button>"
      "  <button type='button'>Register</button>"
      "</div>"
      "<form id='ignored-form'>"
      "  <input type='button' value='Ignore this'>"
      "  <button>Ignore this</button>"
      "  <a id='Submit' value='Ignore this'>"
      "  <div name='BTN'>Ignore this</div>"
      "</form>";

  LoadHTML(kNoFormHtml);
  WebLocalFrame* web_frame = GetMainFrame();
  ASSERT_NE(nullptr, web_frame);
  WebFormElement form_target;
  ASSERT_FALSE(web_frame->GetDocument().Body().IsNull());

  autofill::ButtonTitleList actual =
      GetButtonTitles(form_target, web_frame->GetDocument(), nullptr);

  EXPECT_TRUE(actual.empty());
}

TEST_F(FormAutofillUtilsTest, IsEnabled) {
  LoadHTML(
      "<input type='text' id='name1'>"
      "<input type='password' disabled id='name2'>"
      "<input type='password' id='name3'>"
      "<input type='text' id='name4' disabled>");

  const std::vector<WebElement> dummy_fieldsets;

  WebLocalFrame* web_frame = GetMainFrame();
  ASSERT_TRUE(web_frame);
  std::vector<WebFormControlElement> control_elements;
  WebElementCollection inputs =
      web_frame->GetDocument().GetElementsByHTMLTagName("input");
  for (WebElement element = inputs.FirstItem(); !element.IsNull();
       element = inputs.NextItem()) {
    control_elements.push_back(element.To<WebFormControlElement>());
  }

  std::vector<WebElement> iframe_elements;

  autofill::FormData target;
  EXPECT_TRUE(UnownedFormElementsAndFieldSetsToFormData(
      dummy_fieldsets, control_elements, iframe_elements, nullptr,
      web_frame->GetDocument(), nullptr, EXTRACT_NONE, &target, nullptr));
  const struct {
    const char16_t* const name;
    bool enabled;
  } kExpectedFields[] = {
      {u"name1", true},
      {u"name2", false},
      {u"name3", true},
      {u"name4", false},
  };
  const size_t number_of_cases = base::size(kExpectedFields);
  ASSERT_EQ(number_of_cases, target.fields.size());
  for (size_t i = 0; i < number_of_cases; ++i) {
    EXPECT_EQ(kExpectedFields[i].name, target.fields[i].name);
    EXPECT_EQ(kExpectedFields[i].enabled, target.fields[i].is_enabled);
  }
}

TEST_F(FormAutofillUtilsTest, IsReadonly) {
  LoadHTML(
      "<input type='text' id='name1'>"
      "<input readonly type='password' id='name2'>"
      "<input type='password' id='name3'>"
      "<input type='text' id='name4' readonly>");

  const std::vector<WebElement> dummy_fieldsets;

  WebLocalFrame* web_frame = GetMainFrame();
  ASSERT_TRUE(web_frame);
  std::vector<WebFormControlElement> control_elements;
  WebElementCollection inputs =
      web_frame->GetDocument().GetElementsByHTMLTagName("input");
  for (WebElement element = inputs.FirstItem(); !element.IsNull();
       element = inputs.NextItem()) {
    control_elements.push_back(element.To<WebFormControlElement>());
  }

  std::vector<WebElement> iframe_elements;

  autofill::FormData target;
  EXPECT_TRUE(UnownedFormElementsAndFieldSetsToFormData(
      dummy_fieldsets, control_elements, iframe_elements, nullptr,
      web_frame->GetDocument(), nullptr, EXTRACT_NONE, &target, nullptr));
  const struct {
    const char16_t* const name;
    bool readonly;
  } kExpectedFields[] = {
      {u"name1", false},
      {u"name2", true},
      {u"name3", false},
      {u"name4", true},
  };
  const size_t number_of_cases = base::size(kExpectedFields);
  ASSERT_EQ(number_of_cases, target.fields.size());
  for (size_t i = 0; i < number_of_cases; ++i) {
    EXPECT_EQ(kExpectedFields[i].name, target.fields[i].name);
    EXPECT_EQ(kExpectedFields[i].readonly, target.fields[i].is_readonly);
  }
}

TEST_F(FormAutofillUtilsTest, IsFocusable) {
  LoadHTML(
      "<input type='text' id='name1' value='123'>"
      "<input type='text' id='name2' style='display:none'>");

  const std::vector<WebElement> dummy_fieldsets;

  WebLocalFrame* web_frame = GetMainFrame();
  ASSERT_TRUE(web_frame);

  std::vector<WebFormControlElement> control_elements;
  control_elements.push_back(web_frame->GetDocument()
                                 .GetElementById("name1")
                                 .To<WebFormControlElement>());
  control_elements.push_back(web_frame->GetDocument()
                                 .GetElementById("name2")
                                 .To<WebFormControlElement>());

  EXPECT_TRUE(autofill::form_util::IsWebElementVisible(control_elements[0]));
  EXPECT_FALSE(autofill::form_util::IsWebElementVisible(control_elements[1]));

  std::vector<WebElement> iframe_elements;

  autofill::FormData target;
  EXPECT_TRUE(UnownedFormElementsAndFieldSetsToFormData(
      dummy_fieldsets, control_elements, iframe_elements, nullptr,
      web_frame->GetDocument(), nullptr, EXTRACT_NONE, &target, nullptr));
  ASSERT_EQ(2u, target.fields.size());
  EXPECT_EQ(u"name1", target.fields[0].name);
  EXPECT_TRUE(target.fields[0].is_focusable);
  EXPECT_EQ(u"name2", target.fields[1].name);
  EXPECT_FALSE(target.fields[1].is_focusable);
}

TEST_F(FormAutofillUtilsTest, FindFormByUniqueId) {
  LoadHTML("<body><form id='form1'></form><form id='form2'></form></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  WebVector<WebFormElement> forms = doc.Forms();

  for (const auto& form : forms) {
    EXPECT_EQ(form, FindFormByUniqueRendererId(
                        doc, FormRendererId(form.UniqueRendererFormId())));
  }

  // Expect null form element for non-existing form id.
  FormRendererId non_existing_id(forms[0].UniqueRendererFormId() + 1000);
  EXPECT_TRUE(FindFormByUniqueRendererId(doc, non_existing_id).IsNull());
}

TEST_F(FormAutofillUtilsTest, FindFormControlByUniqueId) {
  LoadHTML(
      "<body><form id='form1'><input id='i1'></form><input id='i2'></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto input1 = doc.GetElementById("i1").To<WebInputElement>();
  auto input2 = doc.GetElementById("i2").To<WebInputElement>();
  FieldRendererId non_existing_id(input2.UniqueRendererFormControlId() + 1000);

  EXPECT_EQ(input1,
            FindFormControlElementByUniqueRendererId(
                doc, FieldRendererId(input1.UniqueRendererFormControlId())));
  EXPECT_EQ(input2,
            FindFormControlElementByUniqueRendererId(
                doc, FieldRendererId(input2.UniqueRendererFormControlId())));
  EXPECT_TRUE(
      FindFormControlElementByUniqueRendererId(doc, non_existing_id).IsNull());
}

TEST_F(FormAutofillUtilsTest, FindFormControlElementsByUniqueIdNoForm) {
  LoadHTML("<body><input id='i1'><input id='i2'><input id='i3'></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto input1 = doc.GetElementById("i1").To<WebInputElement>();
  auto input3 = doc.GetElementById("i3").To<WebInputElement>();
  FieldRendererId non_existing_id(input3.UniqueRendererFormControlId() + 1000);

  std::vector<FieldRendererId> renderer_ids = {
      FieldRendererId(input3.UniqueRendererFormControlId()), non_existing_id,
      FieldRendererId(input1.UniqueRendererFormControlId())};

  auto elements = FindFormControlElementsByUniqueRendererId(doc, renderer_ids);

  ASSERT_EQ(3u, elements.size());
  EXPECT_EQ(input3, elements[0]);
  EXPECT_TRUE(elements[1].IsNull());
  EXPECT_EQ(input1, elements[2]);
}

TEST_F(FormAutofillUtilsTest, FindFormControlElementsByUniqueIdWithForm) {
  LoadHTML(
      "<body><form id='f1'><input id='i1'><input id='i2'></form><input "
      "id='i3'></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto form = doc.GetElementById("f1").To<WebFormElement>();
  auto input1 = doc.GetElementById("i1").To<WebInputElement>();
  auto input3 = doc.GetElementById("i3").To<WebInputElement>();
  FieldRendererId non_existing_id(input3.UniqueRendererFormControlId() + 1000);

  std::vector<FieldRendererId> renderer_ids = {
      FieldRendererId(input3.UniqueRendererFormControlId()), non_existing_id,
      FieldRendererId(input1.UniqueRendererFormControlId())};

  auto elements = FindFormControlElementsByUniqueRendererId(
      doc, FormRendererId(form.UniqueRendererFormId()), renderer_ids);

  // |input3| is not in the form, so it shouldn't be returned.
  ASSERT_EQ(3u, elements.size());
  EXPECT_TRUE(elements[0].IsNull());
  EXPECT_TRUE(elements[1].IsNull());
  EXPECT_EQ(input1, elements[2]);

  // Expect that no elements are returned for non existing form id.
  FormRendererId non_existing_form_id(form.UniqueRendererFormId() + 1000);
  elements = FindFormControlElementsByUniqueRendererId(
      doc, non_existing_form_id, renderer_ids);
  ASSERT_EQ(3u, elements.size());
  EXPECT_TRUE(elements[0].IsNull());
  EXPECT_TRUE(elements[1].IsNull());
  EXPECT_TRUE(elements[2].IsNull());
}

// Tests the extraction of the aria-label attribute.
TEST_F(FormAutofillUtilsTest, GetAriaLabel) {
  LoadHTML("<input id='input' type='text' aria-label='the label'/>");

  WebDocument doc = GetMainFrame()->GetDocument();
  auto element = doc.GetElementById("input").To<WebInputElement>();
  EXPECT_EQ(autofill::form_util::GetAriaLabel(doc, element), u"the label");
}

// Tests that aria-labelledby works. Simple case: only one id referenced.
TEST_F(FormAutofillUtilsTest, GetAriaLabelledBySingle) {
  LoadHTML(
      "<div id='billing'>Billing</div>"
      "<div>"
      "    <div id='name'>Name</div>"
      "    <input id='input' type='text' aria-labelledby='name'/>"
      "</div>");

  WebDocument doc = GetMainFrame()->GetDocument();
  auto element = doc.GetElementById("input").To<WebInputElement>();
  EXPECT_EQ(autofill::form_util::GetAriaLabel(doc, element), u"Name");
}

// Tests that aria-labelledby works: Complex case: multiple ids referenced.
TEST_F(FormAutofillUtilsTest, GetAriaLabelledByMulti) {
  LoadHTML(
      "<div id='billing'>Billing</div>"
      "<div>"
      "    <div id='name'>Name</div>"
      "    <input id='input' type='text' aria-labelledby='billing name'/>"
      "</div>");

  WebDocument doc = GetMainFrame()->GetDocument();
  auto element = doc.GetElementById("input").To<WebInputElement>();
  EXPECT_EQ(autofill::form_util::GetAriaLabel(doc, element), u"Billing Name");
}

// Tests that aria-labelledby takes precedence over aria-label
TEST_F(FormAutofillUtilsTest, GetAriaLabelledByTakesPrecedence) {
  LoadHTML(
      "<div id='billing'>Billing</div>"
      "<div>"
      "    <div id='name'>Name</div>"
      "    <input id='input' type='text' aria-label='ignored' "
      "         aria-labelledby='name'/>"
      "</div>");

  WebDocument doc = GetMainFrame()->GetDocument();
  auto element = doc.GetElementById("input").To<WebInputElement>();
  EXPECT_EQ(autofill::form_util::GetAriaLabel(doc, element), u"Name");
}

// Tests that an invalid aria-labelledby reference gets ignored (as opposed to
// crashing, for example).
TEST_F(FormAutofillUtilsTest, GetAriaLabelledByInvalid) {
  LoadHTML(
      "<div id='billing'>Billing</div>"
      "<div>"
      "    <div id='name'>Name</div>"
      "    <input id='input' type='text' aria-labelledby='div1 div2'/>"
      "</div>");

  WebDocument doc = GetMainFrame()->GetDocument();
  auto element = doc.GetElementById("input").To<WebInputElement>();
  EXPECT_EQ(autofill::form_util::GetAriaLabel(doc, element), u"");
}

// Tests that invalid aria-labelledby references fall back to aria-label.
TEST_F(FormAutofillUtilsTest, GetAriaLabelledByFallback) {
  LoadHTML(
      "<div id='billing'>Billing</div>"
      "<div>"
      "    <div id='name'>Name</div>"
      "    <input id='input' type='text' aria-label='valid' "
      "          aria-labelledby='div1 div2'/>"
      "</div>");

  WebDocument doc = GetMainFrame()->GetDocument();
  auto element = doc.GetElementById("input").To<WebInputElement>();
  EXPECT_EQ(autofill::form_util::GetAriaLabel(doc, element), u"valid");
}

// Tests that aria-describedby works: Simple case: a single id referenced.
TEST_F(FormAutofillUtilsTest, GetAriaDescribedBySingle) {
  LoadHTML(
      "<input id='input' type='text' aria-describedby='div1'/>"
      "<div id='div1'>aria description</div>");

  WebDocument doc = GetMainFrame()->GetDocument();
  auto element = doc.GetElementById("input").To<WebInputElement>();
  EXPECT_EQ(autofill::form_util::GetAriaDescription(doc, element),
            u"aria description");
}

// Tests that aria-describedby works: Complex case: multiple ids referenced.
TEST_F(FormAutofillUtilsTest, GetAriaDescribedByMulti) {
  LoadHTML(
      "<input id='input' type='text' aria-describedby='div1 div2'/>"
      "<div id='div2'>description</div>"
      "<div id='div1'>aria</div>");

  WebDocument doc = GetMainFrame()->GetDocument();
  auto element = doc.GetElementById("input").To<WebInputElement>();
  EXPECT_EQ(autofill::form_util::GetAriaDescription(doc, element),
            u"aria description");
}

// Tests that invalid aria-describedby returns the empty string.
TEST_F(FormAutofillUtilsTest, GetAriaDescribedByInvalid) {
  LoadHTML("<input id='input' type='text' aria-describedby='invalid'/>");

  WebDocument doc = GetMainFrame()->GetDocument();
  auto element = doc.GetElementById("input").To<WebInputElement>();
  EXPECT_EQ(autofill::form_util::GetAriaDescription(doc, element), u"");
}

TEST_F(FormAutofillUtilsTest, IsFormVisible) {
  LoadHTML("<body><form id='form1'><input id='i1'></form></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto form = doc.GetElementById("form1").To<WebFormElement>();
  FormRendererId form_id(form.UniqueRendererFormId());

  EXPECT_TRUE(autofill::form_util::IsFormVisible(GetMainFrame(), form_id));

  // Hide a form.
  form.SetAttribute("style", "display:none");
  EXPECT_FALSE(autofill::form_util::IsFormVisible(GetMainFrame(), form_id));
}

TEST_F(FormAutofillUtilsTest, IsFormControlVisible) {
  LoadHTML("<body><input id='input1'></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto input = doc.GetElementById("input1").To<WebFormControlElement>();
  FieldRendererId input_id(input.UniqueRendererFormControlId());

  EXPECT_TRUE(IsFormControlVisible(GetMainFrame(), input_id));

  // Hide a field.
  input.SetAttribute("style", "display:none");
  EXPECT_FALSE(
      autofill::form_util::IsFormControlVisible(GetMainFrame(), input_id));
}

TEST_F(FormAutofillUtilsTest, IsActionEmptyFalse) {
  LoadHTML(
      "<body><form id='form1' action='done.html'><input "
      "id='i1'></form></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto web_form = doc.GetElementById("form1").To<WebFormElement>();

  FormData form_data;
  ASSERT_TRUE(WebFormElementToFormData(
      web_form, WebFormControlElement(), nullptr /*field_data_manager*/,
      EXTRACT_VALUE, &form_data, nullptr /* FormFieldData */));

  EXPECT_FALSE(form_data.is_action_empty);
}

TEST_F(FormAutofillUtilsTest, IsActionEmptyTrue) {
  LoadHTML("<body><form id='form1'><input id='i1'></form></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto web_form = doc.GetElementById("form1").To<WebFormElement>();

  FormData form_data;
  ASSERT_TRUE(WebFormElementToFormData(
      web_form, WebFormControlElement(), nullptr /*field_data_manager*/,
      EXTRACT_VALUE, &form_data, nullptr /* FormFieldData */));

  EXPECT_TRUE(form_data.is_action_empty);
}

TEST_F(FormAutofillUtilsTest, ExtractBounds) {
  LoadHTML("<body><form id='form1'><input id='i1'></form></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto web_control = doc.GetElementById("i1").To<WebFormControlElement>();

  FormData form_data;
  ASSERT_TRUE(FindFormAndFieldForFormControlElement(
      web_control, nullptr /*field_data_manager*/, EXTRACT_BOUNDS, &form_data,
      nullptr /* FormFieldData */));

  EXPECT_FALSE(form_data.fields.back().bounds.IsEmpty());
}

TEST_F(FormAutofillUtilsTest, NotExtractBounds) {
  LoadHTML("<body><form id='form1'><input id='i1'></form></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto web_control = doc.GetElementById("i1").To<WebFormControlElement>();

  FormData form_data;
  ASSERT_TRUE(FindFormAndFieldForFormControlElement(
      web_control, nullptr /*field_data_manager*/, &form_data,
      nullptr /* FormFieldData */));

  EXPECT_TRUE(form_data.fields.back().bounds.IsEmpty());
}

TEST_F(FormAutofillUtilsTest, ExtractUnownedBounds) {
  LoadHTML("<body><input id='i1'></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto web_control = doc.GetElementById("i1").To<WebFormControlElement>();

  FormData form_data;
  ASSERT_TRUE(FindFormAndFieldForFormControlElement(
      web_control, nullptr /*field_data_manager*/, EXTRACT_BOUNDS, &form_data,
      nullptr /* FormFieldData */));

  EXPECT_FALSE(form_data.fields.back().bounds.IsEmpty());
}

TEST_F(FormAutofillUtilsTest, GetDataListSuggestions) {
  LoadHTML(
      "<body><input list='datalist_id' name='count' id='i1'><datalist "
      "id='datalist_id'><option value='1'><option "
      "value='2'></datalist></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto web_control = doc.GetElementById("i1").To<WebInputElement>();
  std::vector<std::u16string> values;
  std::vector<std::u16string> labels;
  GetDataListSuggestions(web_control, &values, &labels);
  ASSERT_EQ(values.size(), 2u);
  ASSERT_EQ(labels.size(), 2u);
  EXPECT_EQ(values[0], u"1");
  EXPECT_EQ(values[1], u"2");
  EXPECT_EQ(labels[0], u"");
  EXPECT_EQ(labels[1], u"");
}

TEST_F(FormAutofillUtilsTest, GetDataListSuggestionsWithLabels) {
  LoadHTML(
      "<body><input list='datalist_id' name='count' id='i1'><datalist "
      "id='datalist_id'><option value='1'>one</option><option "
      "value='2'>two</option></datalist></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto web_control = doc.GetElementById("i1").To<WebInputElement>();
  std::vector<std::u16string> values;
  std::vector<std::u16string> labels;
  GetDataListSuggestions(web_control, &values, &labels);
  ASSERT_EQ(values.size(), 2u);
  ASSERT_EQ(labels.size(), 2u);
  EXPECT_EQ(values[0], u"1");
  EXPECT_EQ(values[1], u"2");
  EXPECT_EQ(labels[0], u"one");
  EXPECT_EQ(labels[1], u"two");
}

TEST_F(FormAutofillUtilsTest, ExtractDataList) {
  LoadHTML(
      "<body><input list='datalist_id' name='count' id='i1'><datalist "
      "id='datalist_id'><option value='1'>one</option><option "
      "value='2'>two</option></datalist></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto web_control = doc.GetElementById("i1").To<WebInputElement>();
  FormData form_data;
  FormFieldData form_field_data;
  ASSERT_TRUE(FindFormAndFieldForFormControlElement(
      web_control, nullptr /*field_data_manager*/, EXTRACT_DATALIST, &form_data,
      &form_field_data));

  auto& values = form_data.fields.back().datalist_values;
  auto& labels = form_data.fields.back().datalist_labels;
  ASSERT_EQ(values.size(), 2u);
  ASSERT_EQ(labels.size(), 2u);
  EXPECT_EQ(values[0], u"1");
  EXPECT_EQ(values[1], u"2");
  EXPECT_EQ(labels[0], u"one");
  EXPECT_EQ(labels[1], u"two");
  EXPECT_EQ(form_field_data.datalist_values, values);
  EXPECT_EQ(form_field_data.datalist_labels, labels);
}

TEST_F(FormAutofillUtilsTest, NotExtractDataList) {
  LoadHTML(
      "<body><input list='datalist_id' name='count' id='i1'><datalist "
      "id='datalist_id'><option value='1'>one</option><option "
      "value='2'>two</option></datalist></body>");
  WebDocument doc = GetMainFrame()->GetDocument();
  auto web_control = doc.GetElementById("i1").To<WebInputElement>();
  FormData form_data;
  FormFieldData form_field_data;
  ASSERT_TRUE(FindFormAndFieldForFormControlElement(
      web_control, nullptr /*field_data_manager*/, &form_data,
      &form_field_data));

  EXPECT_TRUE(form_data.fields.back().datalist_values.empty());
  EXPECT_TRUE(form_data.fields.back().datalist_labels.empty());
}

// Tests the visibility detection of iframes.
TEST_F(FormAutofillUtilsTest, IsVisibleIframeTest) {
  // Test cases of <iframe> elements with different styles.
  //
  // The `data-[in]visible` attribute represents whether IsVisibleIframe()
  // is expected to classify the iframe as [in]visible.
  //
  // Since IsVisibleIframe() falls short of what the human user will consider
  // visible or invisible, there are false positives and false negatives. For
  // example, IsVisibleIframe() does not check opacity, so <iframe
  // style="opacity: 0.0"> is a false positive (it's visible to
  // IsVisibleIframe() but invisible to the human).
  //
  // The `data-false="{POSITIVE,NEGATIVE}"` attribute indicates whether the test
  // case to be a false positive/negative compared to human visibility
  // perception. In such a case, not meeting the expectation actually indicates
  // an improvement of IsVisibleIframe(), as it means a false positive/negative
  // has been fixed.
  //
  // The sole purpose of the `data-false` attribute is to document this and to
  // print a message when such a test fails.
  LoadHTML(R"(
      <body>
        <iframe srcdoc="<input>" data-visible   style=""></iframe>
        <iframe srcdoc="<input>" data-visible   style="display: block;"></iframe>
        <iframe srcdoc="<input>" data-visible   style="visibility: visible;"></iframe>

        <iframe srcdoc="<input>" data-invisible style="display: none;"></iframe>
        <iframe srcdoc="<input>" data-invisible style="visibility: hidden;"></iframe>
        <div style="display: none;">     <iframe srcdoc="<input>" data-invisible></iframe></div>
        <div style="visibility: hidden;"><iframe srcdoc="<input>" data-invisible></iframe></div>

        <iframe srcdoc="<input>" data-visible   style="width: 15px; height: 15px;"></iframe>
        <iframe srcdoc="<input>" data-invisible style="width: 15px; height:  5px;"></iframe>
        <iframe srcdoc="<input>" data-invisible style="width:  5px; height: 15px;"></iframe>
        <iframe srcdoc="<input>" data-invisible style="width:  5px; height:  5px;"></iframe>

        <iframe srcdoc="<input>" data-invisible style="width: 1px; height: 1px;"></iframe>
        <iframe srcdoc="<input>" data-invisible style="width: 1px; height: 1px; overflow: visible;" data-false="NEGATIVE"></iframe>

        <iframe srcdoc="<input>" data-visible   style="opacity: 0.0;" data-false="POSITIVE"></iframe>
        <iframe srcdoc="<input>" data-visible   style="opacity: 0.0;" data-false="POSITIVE"></iframe>
        <iframe srcdoc="<input>" data-visible   style="position: absolute; clip: rect(0,0,0,0);" data-false="POSITIVE"></iframe>

        <iframe srcdoc="<input>" data-visible   style="width: 100px; height: 100px; position: absolute; left:    -75px;"></iframe>
        <iframe srcdoc="<input>" data-visible   style="width: 100px; height: 100px; position: absolute; top:     -75px;"></iframe>
        <iframe srcdoc="<input>" data-visible   style="width: 100px; height: 100px; position: absolute; left:   -200px;" data-false="POSITIVE"></iframe>
        <iframe srcdoc="<input>" data-visible   style="width: 100px; height: 100px; position: absolute; top:    -200px;" data-false="POSITIVE"></iframe>
        <iframe srcdoc="<input>" data-visible   style="width: 100px; height: 100px; position: absolute; right:  -200px;" data-false="POSITIVE"></iframe>
        <iframe srcdoc="<input>" data-visible   style="width: 100px; height: 100px; position: absolute; bottom: -200px;" data-false="POSITIVE"></iframe>

        <iframe srcdoc="<input>" data-visible   style=""></iframe> <!-- Finish with a visible frame to make sure all <iframe>s have been closed -->

        <div style="width: 10000; height: 10000"></div>
      </body>)");

  // Ensure that Android runs at default page scale.
  web_view_->SetPageScaleFactor(1.0);

  std::vector<WebElement> iframes = [this] {
    WebDocument doc = GetMainFrame()->GetDocument();
    std::vector<WebElement> result;
    WebElementCollection iframes = doc.GetElementsByHTMLTagName("iframe");
    for (WebElement iframe = iframes.FirstItem(); !iframe.IsNull();
         iframe = iframes.NextItem()) {
      result.push_back(iframe);
    }
    return result;
  }();
  ASSERT_GE(iframes.size(), 16u);

  auto RunTestCases = [](const std::vector<WebElement>& iframes) {
    for (WebElement iframe : iframes) {
      gfx::Rect bounds = iframe.BoundsInViewport();
      bool expectation = iframe.HasAttribute("data-visible");
      SCOPED_TRACE(
          testing::Message()
          << "Iframe with style \n  " << iframe.GetAttribute("style").Ascii()
          << "\nwith dimensions w=" << bounds.width()
          << ",h=" << bounds.height() << " and position x=" << bounds.x()
          << ",y=" << bounds.y()
          << (iframe.HasAttribute("data-false") ? "\nwhich used to be a FALSE "
                                                : "")
          << iframe.GetAttribute("data-false").Ascii());
      ASSERT_TRUE(iframe.HasAttribute("data-visible") !=
                  iframe.HasAttribute("data-invisible"));
      EXPECT_EQ(IsVisibleIframe(iframe), expectation);
    }
  };

  RunTestCases(iframes);

  {
    ExecuteJavaScriptForTests(
        "window.scrollTo(document.body.scrollWidth,document.body.scrollHeight)"
        ";");
    content::RunAllTasksUntilIdle();
    SCOPED_TRACE(testing::Message() << "Scrolled to bottom right");
    RunTestCases(iframes);
  }
}

// Tests that `IsDomPredecessor(lhs, rhs, common_ancestor)` holds iff a DOM
// traversal visits the DOM element with ID |lhs| before the one with ID |rhs|,
// where |common_ancestor| is the ID of an ancestor DOM node.
//
// For this test, DOM element IDs should be named so that if X as visited
// before Y, then X.id is lexicographically less than Y.id.
TEST_F(FormAutofillUtilsTest, IsDomPredecessorTest) {
  LoadHTML(R"(
      <body id=0>
        <div id=00>
          <input id=000>
          <input id=001>
          <div id=002>
            <input id=0020>
          </div>
          <div id=003>
            <input id=0030>
          </div>
          <input id=004>
        </div>
        <div id=01>
          <iframe id=010></iframe>
          <input id=011>
        </div>
      </body>)");

  // The parameter type of IsDomPredecessorTest. The attributes are the IDs of
  // the left and right hand side DOM nodes that are to be compared, and some
  // common ancestor of them.
  struct IsDomPredecessorTestParam {
    std::string lhs_id;
    std::string rhs_id;
    std::string common_ancestor_id;
  };
  std::vector<IsDomPredecessorTestParam> test_cases{
      IsDomPredecessorTestParam{"000", "000", "0"},
      IsDomPredecessorTestParam{"001", "001", "0"},
      IsDomPredecessorTestParam{"000", "001", "0"},
      IsDomPredecessorTestParam{"000", "001", "00"},
      IsDomPredecessorTestParam{"000", "0020", "0"},
      IsDomPredecessorTestParam{"000", "0020", "00"},
      IsDomPredecessorTestParam{"000", "004", "0"},
      IsDomPredecessorTestParam{"000", "004", "00"},
      IsDomPredecessorTestParam{"0020", "0030", "0"},
      IsDomPredecessorTestParam{"0020", "0030", "00"},
      IsDomPredecessorTestParam{"0030", "004", "00"},
      IsDomPredecessorTestParam{"000", "010", "0"},
      IsDomPredecessorTestParam{"0030", "010", "0"},
      IsDomPredecessorTestParam{"01", "011", "0"}};

  for (const auto& test : test_cases) {
    SCOPED_TRACE(testing::Message()
                 << "lhs=" << test.lhs_id << " rhs=" << test.rhs_id
                 << " common_ancestor=" << test.common_ancestor_id);
    ASSERT_NE(test.lhs_id, test.common_ancestor_id);
    ASSERT_NE(test.rhs_id, test.common_ancestor_id);
    ASSERT_TRUE(base::StartsWith(test.lhs_id, test.common_ancestor_id));
    ASSERT_TRUE(base::StartsWith(test.rhs_id, test.common_ancestor_id));
    WebDocument doc = GetMainFrame()->GetDocument();
    WebNode lhs = doc.GetElementById(WebString::FromASCII(test.lhs_id));
    WebNode rhs = doc.GetElementById(WebString::FromASCII(test.rhs_id));
    WebNode common_ancestor =
        doc.GetElementById(WebString::FromASCII(test.common_ancestor_id));
    ASSERT_FALSE(lhs.IsNull());
    ASSERT_FALSE(rhs.IsNull());
    ASSERT_FALSE(common_ancestor.IsNull());
    EXPECT_EQ(test.lhs_id < test.rhs_id,
              IsDomPredecessor(lhs, rhs, common_ancestor));
    EXPECT_EQ(test.rhs_id < test.lhs_id,
              IsDomPredecessor(rhs, lhs, common_ancestor));
  }
}

// The DOM ID of an <input> or <iframe>.
struct FieldOrFrame {
  bool is_frame = false;
  const char* id;
};

// A FieldFramesTest test case contains HTML code. The form with DOM ID
// |form_id| (nullptr for the synthetic form) shall be extracted and its fields
// and frames shall match |fields_and_frames|.
struct FieldFramesTestParam {
  std::string html;
  const char* form_id;
  std::vector<FieldOrFrame> fields_and_frames;
};

class FieldFramesTest
    : public content::RenderViewTest,
      public testing::WithParamInterface<FieldFramesTestParam> {
 public:
  FieldFramesTest() {
    scoped_feature_list_.InitAndEnableFeature(features::kAutofillAcrossIframes);
  }
  ~FieldFramesTest() override = default;

 private:
  base::test::ScopedFeatureList scoped_feature_list_;
};

// Tests that FormData::fields and FormData::child_frames are extracted fully
// and in the correct relative order.
TEST_P(FieldFramesTest, ExtractFieldsAndFrames) {
  FieldFramesTestParam test_case = GetParam();
  SCOPED_TRACE(testing::Message() << "HTML: " << test_case.html);
  LoadHTML(test_case.html.c_str());
  WebDocument doc = GetMainFrame()->GetDocument();

  // Extract the |form_data|.
  FormData form_data;
  FormRendererId host_form;
  if (!test_case.form_id) {  // Synthetic form.
    std::vector<blink::WebElement> fieldsets;
    std::vector<WebFormControlElement> control_elements =
        GetUnownedAutofillableFormFieldElements(doc, &fieldsets);
    std::vector<WebElement> iframe_elements =
        form_util::GetUnownedIframeElements(doc);
    ASSERT_TRUE(UnownedFormElementsAndFieldSetsToFormData(
        fieldsets, control_elements, iframe_elements, nullptr, doc, nullptr,
        EXTRACT_NONE, &form_data, nullptr));
    host_form = FormRendererId();
  } else {  // Real <form>.
    ASSERT_GT(std::strlen(test_case.form_id), 0u);
    auto form_element =
        doc.GetElementById(blink::WebString::FromASCII(test_case.form_id))
            .To<WebFormElement>();
    ASSERT_TRUE(WebFormElementToFormData(form_element, WebFormControlElement(),
                                         nullptr, EXTRACT_NONE, &form_data,
                                         nullptr));
    host_form = FormRendererId(form_element.UniqueRendererFormId());
  }

  // Check that all fields and iframes were extracted.
  EXPECT_EQ(form_data.fields.size() + form_data.child_frames.size(),
            test_case.fields_and_frames.size());

  // Check that all fields were extracted. Do so by checking for each |field| in
  // `test_case.fields_and_frames` that the DOM element with ID `field.id`
  // corresponds to the next (`i`th) field in `form_data.fields`.
  size_t i = 0;
  for (const FieldOrFrame& field : test_case.fields_and_frames) {
    if (field.is_frame)
      continue;
    SCOPED_TRACE(testing::Message() << "Checking the " << i
                                    << "th field (id = " << field.id << ")");
    WebElement element =
        doc.GetElementById(blink::WebString::FromASCII(field.id));
    ASSERT_FALSE(element.IsNull());
    ASSERT_TRUE(element.IsFormControlElement());
    WebFormControlElement control = element.To<WebFormControlElement>();
    ASSERT_FALSE(control.IsNull());
    FieldRendererId renderer_id(control.UniqueRendererFormControlId());
    EXPECT_EQ(form_data.fields[i].host_form_id, host_form);
    EXPECT_EQ(form_data.fields[i].unique_renderer_id, renderer_id);
    ++i;
  }

  // Check that all frames were extracted into `form_data.child_frames`
  // analogously to the above loop for `form_data.fields`.
  //
  // In addition, check that `form_data.child_frames[i].predecessor` encodes the
  // correct ordering, i.e., that `form_data.child_frames[i].predecessor` is the
  // index of the last field in `form_data.fields` that precedes the `i`th frame
  // in `form_data.child_frames`.
  i = 0;
  int preceding_field_index = -1;
  for (const auto& frame : test_case.fields_and_frames) {
    if (!frame.is_frame) {
      ++preceding_field_index;
      continue;
    }
    SCOPED_TRACE(testing::Message() << "Checking the " << i
                                    << "th frame (id = " << frame.id << ")");
    WebElement element =
        doc.GetElementById(blink::WebString::FromASCII(frame.id));
    ASSERT_FALSE(element.IsNull());
    ASSERT_TRUE(element.HasHTMLTagName("iframe"));
    auto is_empty = [](auto token) { return token.is_empty(); };
    FrameToken frame_token = GetFrameToken(element);
    EXPECT_FALSE(absl::visit(is_empty, form_data.child_frames[i].token));
    EXPECT_EQ(form_data.child_frames[i].token, frame_token);
    EXPECT_EQ(form_data.child_frames[i].predecessor, preceding_field_index);
    ++i;
  }
}

// Creates 32 test cases containing forms which differ in five bits: whether or
// not the form of interest is a synthetic form, and whether the first, second,
// third, and fourth element are a form control field or an iframe. This form is
// additionally surrounded by two other forms before and after itself. An
// example:
//   <body>
//     <!-- Two forms not of interest follow -->
//     <form><input><iframe></iframe></form>
//     <input><iframe></iframe>
//     <!-- The form of interest follows -->
//     <form id='MY_FORM_ID'>
//       <input>
//       <input>
//       <iframe></iframe>
//       <iframe></iframe>
//     </form>
//     <!-- Two forms not of interest follow -->
//     <form><input><iframe></iframe></form>
//     <input><iframe></iframe>
//   </body>
INSTANTIATE_TEST_SUITE_P(
    FormAutofillUtilsTest,
    FieldFramesTest,
    testing::ValuesIn([] {
      // Creates a FieldFramesTestParam. The fifth bit encodes whether the form
      // is a synthetic form or not, and the first four bits encode whether its
      // four elements are fields or frames.
      //
      // The choice of four is to cover multiple elements of the same kind
      // following each other and being surrounded by other fields, e.g.,
      // <input><iframe><iframe><input>.
      auto MakeTestCase = [](std::bitset<5> bitset) {
        std::vector<FieldOrFrame> fields_and_frames{
            {.is_frame = bitset.test(0), .id = "0"},
            {.is_frame = bitset.test(1), .id = "1"},
            {.is_frame = bitset.test(2), .id = "2"},
            {.is_frame = bitset.test(3), .id = "3"},
        };
        bool is_synthetic_form = bitset.test(4);
        const char* form_id = is_synthetic_form ? nullptr : "MY_FORM_ID";

        // Create a HTML page according to |is_synthetic_form| and
        // |fields_and_frames|: it contains four <input> or <iframe> elements,
        // potentially contained in a <form>. Additionally, before and after
        // this form, it contains some other <input> and <iframe> elements that
        // do not belong to the form of interest.
        std::string html;
        for (const FieldOrFrame& field_or_frame : fields_and_frames) {
          html += base::StringPrintf(field_or_frame.is_frame
                                         ? "<iframe id='%s'></iframe>"
                                         : "<input id='%s'>",
                                     field_or_frame.id);
        }
        if (!is_synthetic_form) {
          html = base::StringPrintf("<form id='%s'>%s</form>", form_id,
                                    html.c_str());
          const char* other_forms =
              "<input><iframe></iframe> <form><input><iframe></iframe></form>";
          html = base::StrCat({other_forms, html, other_forms});
        } else {
          const char* other_form = "<form><input><iframe></iframe></form>";
          html = base::StrCat({other_form, html, other_form});
        }
        html = base::StringPrintf("<body>%s</body>", html.c_str());
        return FieldFramesTestParam{.html = html,
                                    .form_id = form_id,
                                    .fields_and_frames = fields_and_frames};
      };

      // Create all combinations of test cases.
      std::vector<FieldFramesTestParam> cases;
      for (uint64_t bitset = 0; bitset < (1 << 5); ++bitset)
        cases.push_back(MakeTestCase(std::bitset<5>(bitset)));

      // Check that we have 32 distinct test cases.
      DCHECK_EQ(cases.size(), 32u);
      DCHECK(base::ranges::all_of(cases, [&](const auto& case1) {
        return base::ranges::all_of(cases, [&](const auto& case2) {
          return &case1 == &case2 || case1.html != case2.html;
        });
      }));
      return cases;
    }()));

// Tests that if the number of iframes exceeds |kMaxParseableFrames|, neither
// fields nor child frames of that form are extracted.
TEST_F(FormAutofillUtilsTest, ExtractNoFieldsOrFramesIfTooManyIframes) {
  base::test::ScopedFeatureList scoped_feature_list;
  scoped_feature_list.InitAndEnableFeature(features::kAutofillAcrossIframes);

  auto CreateFormElement = [this](const char* element) {
    std::string js = base::StringPrintf(
        "document.forms[0].appendChild(document.createElement('%s'))", element);
    ExecuteJavaScriptForTests(js.c_str());
  };

  LoadHTML(R"(<html><body><form id='f'></form>)");
  for (size_t i = 0; i < kMaxParseableFields - 1; ++i)
    CreateFormElement("input");
  for (size_t i = 0; i < kMaxParseableFrames - 1; ++i)
    CreateFormElement("iframe");

  // Ensure that Android runs at default page scale.
  web_view_->SetPageScaleFactor(1.0);

  WebDocument doc = GetMainFrame()->GetDocument();
  WebFormElement form =
      doc.GetElementById(blink::WebString::FromASCII("f")).To<WebFormElement>();
  {
    FormData form_data;
    ASSERT_TRUE(WebFormElementToFormData(form, WebFormControlElement(), nullptr,
                                         EXTRACT_NONE, &form_data, nullptr));
    EXPECT_EQ(form_data.fields.size(), kMaxParseableFields - 1);
    EXPECT_EQ(form_data.child_frames.size(), kMaxParseableFrames - 1);
  }

  // There may be multiple checks (e.g., == kMaxParseableFrames, <=
  // kMaxParseableFrames, < kMaxParseableFrames), so we test different numbers
  // of <iframe> elements.
  for (int i = 0; i < 3; ++i) {
    CreateFormElement("iframe");
    FormData form_data;
    ASSERT_FALSE(WebFormElementToFormData(form, WebFormControlElement(),
                                          nullptr, EXTRACT_NONE, &form_data,
                                          nullptr));
    EXPECT_TRUE(form_data.fields.empty());
    EXPECT_TRUE(form_data.child_frames.empty());
  }
}

// Tests that if the number of fields exceeds |kMaxParseableFields|, neither
// fields nor child frames of that form are extracted.
TEST_F(FormAutofillUtilsTest, ExtractNoFieldsOrFramesIfTooManyFields) {
  base::test::ScopedFeatureList scoped_feature_list;
  scoped_feature_list.InitAndEnableFeature(features::kAutofillAcrossIframes);

  auto CreateFormElement = [this](const char* element) {
    std::string js = base::StringPrintf(
        "document.forms[0].appendChild(document.createElement('%s'))", element);
    ExecuteJavaScriptForTests(js.c_str());
  };

  LoadHTML(R"(<html><body><form id='f'></form>)");
  for (size_t i = 0; i < kMaxParseableFields - 1; ++i)
    CreateFormElement("input");
  for (size_t i = 0; i < kMaxParseableFrames - 1; ++i)
    CreateFormElement("iframe");

  // Ensure that Android runs at default page scale.
  web_view_->SetPageScaleFactor(1.0);

  WebDocument doc = GetMainFrame()->GetDocument();
  WebFormElement form =
      doc.GetElementById(blink::WebString::FromASCII("f")).To<WebFormElement>();
  {
    FormData form_data;
    ASSERT_TRUE(WebFormElementToFormData(form, WebFormControlElement(), nullptr,
                                         EXTRACT_NONE, &form_data, nullptr));
    EXPECT_EQ(form_data.fields.size(), kMaxParseableFields - 1);
    EXPECT_EQ(form_data.child_frames.size(), kMaxParseableFrames - 1);
  }

  // There may be multiple checks (e.g., == kMaxParseableFields, <=
  // kMaxParseableFields, < kMaxParseableFields), so we test different numbers
  // of <input> elements.
  for (int i = 0; i < 3; ++i) {
    SCOPED_TRACE(base::NumberToString(i));
    CreateFormElement("input");
    FormData form_data;
    ASSERT_FALSE(WebFormElementToFormData(form, WebFormControlElement(),
                                          nullptr, EXTRACT_NONE, &form_data,
                                          nullptr));
    EXPECT_TRUE(form_data.fields.empty());
    EXPECT_TRUE(form_data.child_frames.empty());
  }
}

}  // namespace
}  // namespace form_util
}  // namespace autofill
