/*
 * Copyright (C) 2005-2024 Apple Inc. All rights reserved.
 * Copyright (C) 2006 Jon Shier (jshier@iastate.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#pragma once

#include "ThreadGlobalData.h"
#include <array>
#include <functional>
#include <wtf/text/AtomString.h>

namespace WebCore {

struct EventNames {
    WTF_MAKE_NONCOPYABLE(EventNames); WTF_MAKE_FAST_ALLOCATED;

public:
    const AtomString DOMActivateEvent;
    const AtomString DOMCharacterDataModifiedEvent;
    const AtomString DOMContentLoadedEvent;
    const AtomString DOMNodeInsertedEvent;
    const AtomString DOMNodeInsertedIntoDocumentEvent;
    const AtomString DOMNodeRemovedEvent;
    const AtomString DOMNodeRemovedFromDocumentEvent;
    const AtomString DOMSubtreeModifiedEvent;
    const AtomString abortEvent;
    const AtomString activateEvent;
    const AtomString activeEvent;
    const AtomString addsourcebufferEvent;
    const AtomString addstreamEvent;
    const AtomString addtrackEvent;
    const AtomString afterprintEvent;
    const AtomString animationcancelEvent;
    const AtomString animationendEvent;
    const AtomString animationiterationEvent;
    const AtomString animationstartEvent;
    const AtomString audioendEvent;
    const AtomString audioprocessEvent;
    const AtomString audiostartEvent;
    const AtomString autocompleteEvent;
    const AtomString autocompleteerrorEvent;
    const AtomString backgroundfetchabortEvent;
    const AtomString backgroundfetchclickEvent;
    const AtomString backgroundfetchfailEvent;
    const AtomString backgroundfetchsuccessEvent;
    const AtomString beforecopyEvent;
    const AtomString beforecutEvent;
    const AtomString beforeinputEvent;
    const AtomString beforeloadEvent;
    const AtomString beforepasteEvent;
    const AtomString beforeprintEvent;
    const AtomString beforetoggleEvent;
    const AtomString beforeunloadEvent;
    const AtomString beginEventEvent;
    const AtomString blockedEvent;
    const AtomString blurEvent;
    const AtomString boundaryEvent;
    const AtomString bufferedamountlowEvent;
    const AtomString bufferedchangeEvent;
    const AtomString cachedEvent;
    const AtomString cancelEvent;
    const AtomString canplayEvent;
    const AtomString canplaythroughEvent;
    const AtomString changeEvent;
    const AtomString chargingchangeEvent;
    const AtomString chargingtimechangeEvent;
    const AtomString checkingEvent;
    const AtomString clickEvent;
    const AtomString closeEvent;
    const AtomString closingEvent;
    const AtomString completeEvent;
    const AtomString compositionendEvent;
    const AtomString compositionstartEvent;
    const AtomString compositionupdateEvent;
    const AtomString configurationchangeEvent;
    const AtomString connectEvent;
    const AtomString connectingEvent;
    const AtomString connectionstatechangeEvent;
    const AtomString contentvisibilityautostatechangeEvent;
    const AtomString contextmenuEvent;
    const AtomString controllerchangeEvent;
    const AtomString cookiechangeEvent;
    const AtomString coordinatorstatechangeEvent;
    const AtomString copyEvent;
#if ENABLE(APPLE_PAY_COUPON_CODE)
    const AtomString couponcodechangedEvent;
#endif
    const AtomString cuechangeEvent;
    const AtomString currententrychangeEvent;
    const AtomString cutEvent;
    const AtomString dataavailableEvent;
    const AtomString datachannelEvent;
    const AtomString dblclickEvent;
    const AtomString dequeueEvent;
    const AtomString devicechangeEvent;
    const AtomString devicemotionEvent;
    const AtomString deviceorientationEvent;
    const AtomString dischargingtimechangeEvent;
    const AtomString disconnectEvent;
    const AtomString disposeEvent;
    const AtomString downloadingEvent;
    const AtomString dragEvent;
    const AtomString dragendEvent;
    const AtomString dragenterEvent;
    const AtomString dragleaveEvent;
    const AtomString dragoverEvent;
    const AtomString dragstartEvent;
    const AtomString dropEvent;
    const AtomString durationchangeEvent;
    const AtomString emptiedEvent;
    const AtomString encryptedEvent;
    const AtomString endEvent;
    const AtomString endEventEvent;
    const AtomString endedEvent;
    const AtomString endstreamingEvent;
    const AtomString enterEvent;
    const AtomString enterpictureinpictureEvent;
    const AtomString errorEvent;
    const AtomString exitEvent;
    const AtomString fetchEvent;
    const AtomString finishEvent;
    const AtomString focusEvent;
    const AtomString focusinEvent;
    const AtomString focusoutEvent;
    const AtomString formdataEvent;
    const AtomString fullscreenchangeEvent;
    const AtomString fullscreenerrorEvent;
    const AtomString gamepadconnectedEvent;
    const AtomString gamepaddisconnectedEvent;
    const AtomString gatheringstatechangeEvent;
    const AtomString gesturechangeEvent;
    const AtomString gestureendEvent;
    const AtomString gesturescrollendEvent;
    const AtomString gesturescrollstartEvent;
    const AtomString gesturescrollupdateEvent;
    const AtomString gesturestartEvent;
    const AtomString gesturetapEvent;
    const AtomString gesturetapdownEvent;
    const AtomString gotpointercaptureEvent;
    const AtomString hashchangeEvent;
    const AtomString icecandidateEvent;
    const AtomString icecandidateerrorEvent;
    const AtomString iceconnectionstatechangeEvent;
    const AtomString icegatheringstatechangeEvent;
    const AtomString inactiveEvent;
    const AtomString inputEvent;
    const AtomString inputsourceschangeEvent;
    const AtomString installEvent;
    const AtomString invalidEvent;
    const AtomString invokeEvent;
    const AtomString keydownEvent;
    const AtomString keypressEvent;
    const AtomString keystatuseschangeEvent;
    const AtomString keyupEvent;
    const AtomString languagechangeEvent;
    const AtomString leavepictureinpictureEvent;
    const AtomString levelchangeEvent;
    const AtomString loadEvent;
    const AtomString loadeddataEvent;
    const AtomString loadedmetadataEvent;
    const AtomString loadendEvent;
    const AtomString loadingEvent;
    const AtomString loadingdoneEvent;
    const AtomString loadingerrorEvent;
    const AtomString loadstartEvent;
    const AtomString lostpointercaptureEvent;
    const AtomString markEvent;
    const AtomString merchantvalidationEvent;
    const AtomString messageEvent;
    const AtomString messageerrorEvent;
    const AtomString mousedownEvent;
    const AtomString mouseenterEvent;
    const AtomString mouseleaveEvent;
    const AtomString mousemoveEvent;
    const AtomString mouseoutEvent;
    const AtomString mouseoverEvent;
    const AtomString mouseupEvent;
    const AtomString mousewheelEvent;
    const AtomString muteEvent;
    const AtomString navigateEvent;
    const AtomString navigateerrorEvent;
    const AtomString navigatesuccessEvent;
    const AtomString negotiationneededEvent;
    const AtomString nexttrackEvent;
    const AtomString nomatchEvent;
    const AtomString notificationclickEvent;
    const AtomString notificationcloseEvent;
    const AtomString noupdateEvent;
    const AtomString obsoleteEvent;
    const AtomString offlineEvent;
    const AtomString onlineEvent;
    const AtomString openEvent;
    const AtomString orientationchangeEvent;
    const AtomString overflowchangedEvent;
    const AtomString pagehideEvent;
    const AtomString pageshowEvent;
    const AtomString pasteEvent;
    const AtomString pauseEvent;
    const AtomString payerdetailchangeEvent;
    const AtomString paymentauthorizedEvent;
    const AtomString paymentmethodchangeEvent;
    const AtomString paymentmethodselectedEvent;
    const AtomString playEvent;
    const AtomString playingEvent;
    const AtomString pointercancelEvent;
    const AtomString pointerdownEvent;
    const AtomString pointerenterEvent;
    const AtomString pointerleaveEvent;
    const AtomString pointerlockchangeEvent;
    const AtomString pointerlockerrorEvent;
    const AtomString pointermoveEvent;
    const AtomString pointeroutEvent;
    const AtomString pointeroverEvent;
    const AtomString pointerupEvent;
    const AtomString popstateEvent;
    const AtomString previoustrackEvent;
    const AtomString processorerrorEvent;
    const AtomString progressEvent;
    const AtomString pushEvent;
    const AtomString pushnotificationEvent;
    const AtomString pushsubscriptionchangeEvent;
    const AtomString qualitychangeEvent;
    const AtomString ratechangeEvent;
    const AtomString readystatechangeEvent;
    const AtomString rejectionhandledEvent;
    const AtomString releaseEvent;
    const AtomString removeEvent;
    const AtomString removesourcebufferEvent;
    const AtomString removestreamEvent;
    const AtomString removetrackEvent;
    const AtomString resetEvent;
    const AtomString resizeEvent;
    const AtomString resourcetimingbufferfullEvent;
    const AtomString resultEvent;
    const AtomString resumeEvent;
    const AtomString rtctransformEvent;
    const AtomString scrollEvent;
    const AtomString searchEvent;
    const AtomString securitypolicyviolationEvent;
    const AtomString seekedEvent;
    const AtomString seekingEvent;
    const AtomString selectEvent;
    const AtomString selectedcandidatepairchangeEvent;
    const AtomString selectendEvent;
    const AtomString selectionchangeEvent;
    const AtomString selectstartEvent;
    const AtomString shippingaddresschangeEvent;
    const AtomString shippingcontactselectedEvent;
    const AtomString shippingmethodselectedEvent;
    const AtomString shippingoptionchangeEvent;
    const AtomString showEvent;
    const AtomString signalingstatechangeEvent;
    const AtomString slotchangeEvent;
    const AtomString soundendEvent;
    const AtomString soundstartEvent;
    const AtomString sourcecloseEvent;
    const AtomString sourceendedEvent;
    const AtomString sourceopenEvent;
    const AtomString speechendEvent;
    const AtomString speechstartEvent;
    const AtomString squeezeEvent;
    const AtomString squeezeendEvent;
    const AtomString squeezestartEvent;
    const AtomString stalledEvent;
    const AtomString startEvent;
    const AtomString startedEvent;
    const AtomString startstreamingEvent;
    const AtomString statechangeEvent;
    const AtomString stopEvent;
    const AtomString storageEvent;
    const AtomString submitEvent;
    const AtomString successEvent;
    const AtomString suspendEvent;
    const AtomString textInputEvent;
    const AtomString timeoutEvent;
    const AtomString timeupdateEvent;
    const AtomString toggleEvent;
    const AtomString tonechangeEvent;
    const AtomString touchcancelEvent;
    const AtomString touchendEvent;
    const AtomString touchforcechangeEvent;
    const AtomString touchmoveEvent;
    const AtomString touchstartEvent;
    const AtomString trackEvent;
    const AtomString transitioncancelEvent;
    const AtomString transitionendEvent;
    const AtomString transitionrunEvent;
    const AtomString transitionstartEvent;
    const AtomString uncapturederrorEvent;
    const AtomString unhandledrejectionEvent;
    const AtomString unloadEvent;
    const AtomString unmuteEvent;
    const AtomString updateEvent;
    const AtomString updateendEvent;
    const AtomString updatefoundEvent;
    const AtomString updatereadyEvent;
    const AtomString updatestartEvent;
    const AtomString upgradeneededEvent;
    const AtomString validatemerchantEvent;
    const AtomString versionchangeEvent;
    const AtomString visibilitychangeEvent;
    const AtomString voiceschangedEvent;
    const AtomString volumechangeEvent;
    const AtomString waitingEvent;
    const AtomString waitingforkeyEvent;
    const AtomString webglcontextcreationerrorEvent;
    const AtomString webglcontextlostEvent;
    const AtomString webglcontextrestoredEvent;
    const AtomString webkitAnimationEndEvent;
    const AtomString webkitAnimationIterationEvent;
    const AtomString webkitAnimationStartEvent;
    const AtomString webkitBeforeTextInsertedEvent;
    const AtomString webkitTransitionEndEvent;
    const AtomString webkitbeginfullscreenEvent;
    const AtomString webkitcurrentplaybacktargetiswirelesschangedEvent;
    const AtomString webkitendfullscreenEvent;
    const AtomString webkitfullscreenchangeEvent;
    const AtomString webkitfullscreenerrorEvent;
    const AtomString webkitkeyaddedEvent;
    const AtomString webkitkeyerrorEvent;
    const AtomString webkitkeymessageEvent;
    const AtomString webkitmouseforcechangedEvent;
    const AtomString webkitmouseforcedownEvent;
    const AtomString webkitmouseforceupEvent;
    const AtomString webkitmouseforcewillbeginEvent;
    const AtomString webkitneedkeyEvent;
    const AtomString webkitnetworkinfochangeEvent;
    const AtomString webkitplaybacktargetavailabilitychangedEvent;
    const AtomString webkitpresentationmodechangedEvent;
    const AtomString webkitremovesourcebufferEvent;
    const AtomString webkitsourcecloseEvent;
    const AtomString webkitsourceendedEvent;
    const AtomString webkitsourceopenEvent;
    const AtomString wheelEvent;
    const AtomString writeEvent;
    const AtomString writeendEvent;
    const AtomString writestartEvent;
    const AtomString zoomEvent;

    template<class... Args>
    static std::unique_ptr<EventNames> create(Args&&... args)
    {
        return std::unique_ptr<EventNames>(new EventNames(std::forward<Args>(args)...));
    }

    inline bool isCSSAnimationEventType(const AtomString&) const;
    inline std::array<const AtomString, 7> CSSAnimationEventNames() const;

    inline bool isCSSTransitionEventType(const AtomString&) const;
    inline std::array<const AtomString, 5> CSSTransitionEventNames() const;

    inline bool isExtendedTouchRelatedEventType(const AtomString&) const;
    inline std::array<const AtomString, 16> extendedTouchRelatedEventNames() const;

    inline bool isGamepadEventType(const AtomString&) const;
    inline std::array<const AtomString, 2> gamepadEventNames() const;

    inline bool isGestureEventType(const AtomString&) const;
    inline std::array<const AtomString, 3> gestureEventNames() const;

    inline bool isMouseClickRelatedEventType(const AtomString&) const;
    inline std::array<const AtomString, 4> mouseClickRelatedEventNames() const;

    inline bool isMouseMoveRelatedEventType(const AtomString&) const;
    inline std::array<const AtomString, 3> mouseMoveRelatedEventNames() const;

    inline bool isSimulatedMouseEventType(const AtomString&) const;
    inline std::array<const AtomString, 3> simulatedMouseEventNames() const;

    inline bool isTouchRelatedEventType(const AtomString&) const;
    inline std::array<const AtomString, 13> touchRelatedEventNames() const;

    inline bool isTouchScrollBlockingEventType(const AtomString&) const;
    inline std::array<const AtomString, 2> touchScrollBlockingEventNames() const;

    inline bool isWheelEventType(const AtomString&) const;
    inline std::array<const AtomString, 2> wheelEventNames() const;

    inline std::array<const AtomString, 319> allEventNames() const;

private:
    EventNames();
};

const EventNames& eventNames();

inline const EventNames& eventNames()
{
    return threadGlobalData().eventNames();
}

inline bool EventNames::isCSSAnimationEventType(const AtomString& type) const
{
    return type == animationcancelEvent
        || type == animationendEvent
        || type == animationiterationEvent
        || type == animationstartEvent
        || type == webkitAnimationEndEvent
        || type == webkitAnimationIterationEvent
        || type == webkitAnimationStartEvent;
}

inline std::array<const AtomString, 7> EventNames::CSSAnimationEventNames() const
{
    return { {
        animationcancelEvent,
        animationendEvent,
        animationiterationEvent,
        animationstartEvent,
        webkitAnimationEndEvent,
        webkitAnimationIterationEvent,
        webkitAnimationStartEvent,
    } };
}

inline bool EventNames::isCSSTransitionEventType(const AtomString& type) const
{
    return type == transitioncancelEvent
        || type == transitionendEvent
        || type == transitionrunEvent
        || type == transitionstartEvent
        || type == webkitTransitionEndEvent;
}

inline std::array<const AtomString, 5> EventNames::CSSTransitionEventNames() const
{
    return { {
        transitioncancelEvent,
        transitionendEvent,
        transitionrunEvent,
        transitionstartEvent,
        webkitTransitionEndEvent,
    } };
}

inline bool EventNames::isExtendedTouchRelatedEventType(const AtomString& type) const
{
    return type == mousedownEvent
        || type == mousemoveEvent
        || type == mouseupEvent
        || type == pointercancelEvent
        || type == pointerdownEvent
        || type == pointerenterEvent
        || type == pointerleaveEvent
        || type == pointermoveEvent
        || type == pointeroutEvent
        || type == pointeroverEvent
        || type == pointerupEvent
        || type == touchcancelEvent
        || type == touchendEvent
        || type == touchforcechangeEvent
        || type == touchmoveEvent
        || type == touchstartEvent;
}

inline std::array<const AtomString, 16> EventNames::extendedTouchRelatedEventNames() const
{
    return { {
        mousedownEvent,
        mousemoveEvent,
        mouseupEvent,
        pointercancelEvent,
        pointerdownEvent,
        pointerenterEvent,
        pointerleaveEvent,
        pointermoveEvent,
        pointeroutEvent,
        pointeroverEvent,
        pointerupEvent,
        touchcancelEvent,
        touchendEvent,
        touchforcechangeEvent,
        touchmoveEvent,
        touchstartEvent,
    } };
}

inline bool EventNames::isGamepadEventType(const AtomString& type) const
{
    return type == gamepadconnectedEvent
        || type == gamepaddisconnectedEvent;
}

inline std::array<const AtomString, 2> EventNames::gamepadEventNames() const
{
    return { {
        gamepadconnectedEvent,
        gamepaddisconnectedEvent,
    } };
}

inline bool EventNames::isGestureEventType(const AtomString& type) const
{
    return type == gesturechangeEvent
        || type == gestureendEvent
        || type == gesturestartEvent;
}

inline std::array<const AtomString, 3> EventNames::gestureEventNames() const
{
    return { {
        gesturechangeEvent,
        gestureendEvent,
        gesturestartEvent,
    } };
}

inline bool EventNames::isMouseClickRelatedEventType(const AtomString& type) const
{
    return type == DOMActivateEvent
        || type == clickEvent
        || type == mousedownEvent
        || type == mouseupEvent;
}

inline std::array<const AtomString, 4> EventNames::mouseClickRelatedEventNames() const
{
    return { {
        DOMActivateEvent,
        clickEvent,
        mousedownEvent,
        mouseupEvent,
    } };
}

inline bool EventNames::isMouseMoveRelatedEventType(const AtomString& type) const
{
    return type == mousemoveEvent
        || type == mouseoutEvent
        || type == mouseoverEvent;
}

inline std::array<const AtomString, 3> EventNames::mouseMoveRelatedEventNames() const
{
    return { {
        mousemoveEvent,
        mouseoutEvent,
        mouseoverEvent,
    } };
}

inline bool EventNames::isSimulatedMouseEventType(const AtomString& type) const
{
    return type == mousedownEvent
        || type == mousemoveEvent
        || type == mouseupEvent;
}

inline std::array<const AtomString, 3> EventNames::simulatedMouseEventNames() const
{
    return { {
        mousedownEvent,
        mousemoveEvent,
        mouseupEvent,
    } };
}

inline bool EventNames::isTouchRelatedEventType(const AtomString& type) const
{
    return type == pointercancelEvent
        || type == pointerdownEvent
        || type == pointerenterEvent
        || type == pointerleaveEvent
        || type == pointermoveEvent
        || type == pointeroutEvent
        || type == pointeroverEvent
        || type == pointerupEvent
        || type == touchcancelEvent
        || type == touchendEvent
        || type == touchforcechangeEvent
        || type == touchmoveEvent
        || type == touchstartEvent;
}

inline std::array<const AtomString, 13> EventNames::touchRelatedEventNames() const
{
    return { {
        pointercancelEvent,
        pointerdownEvent,
        pointerenterEvent,
        pointerleaveEvent,
        pointermoveEvent,
        pointeroutEvent,
        pointeroverEvent,
        pointerupEvent,
        touchcancelEvent,
        touchendEvent,
        touchforcechangeEvent,
        touchmoveEvent,
        touchstartEvent,
    } };
}

inline bool EventNames::isTouchScrollBlockingEventType(const AtomString& type) const
{
    return type == touchmoveEvent
        || type == touchstartEvent;
}

inline std::array<const AtomString, 2> EventNames::touchScrollBlockingEventNames() const
{
    return { {
        touchmoveEvent,
        touchstartEvent,
    } };
}

inline bool EventNames::isWheelEventType(const AtomString& type) const
{
    return type == mousewheelEvent
        || type == wheelEvent;
}

inline std::array<const AtomString, 2> EventNames::wheelEventNames() const
{
    return { {
        mousewheelEvent,
        wheelEvent,
    } };
}

inline std::array<const AtomString, 319> EventNames::allEventNames() const
{
    return { {
        DOMActivateEvent,
        DOMCharacterDataModifiedEvent,
        DOMContentLoadedEvent,
        DOMNodeInsertedEvent,
        DOMNodeInsertedIntoDocumentEvent,
        DOMNodeRemovedEvent,
        DOMNodeRemovedFromDocumentEvent,
        DOMSubtreeModifiedEvent,
        abortEvent,
        activateEvent,
        activeEvent,
        addsourcebufferEvent,
        addstreamEvent,
        addtrackEvent,
        afterprintEvent,
        animationcancelEvent,
        animationendEvent,
        animationiterationEvent,
        animationstartEvent,
        audioendEvent,
        audioprocessEvent,
        audiostartEvent,
        autocompleteEvent,
        autocompleteerrorEvent,
        backgroundfetchabortEvent,
        backgroundfetchclickEvent,
        backgroundfetchfailEvent,
        backgroundfetchsuccessEvent,
        beforecopyEvent,
        beforecutEvent,
        beforeinputEvent,
        beforeloadEvent,
        beforepasteEvent,
        beforeprintEvent,
        beforetoggleEvent,
        beforeunloadEvent,
        beginEventEvent,
        blockedEvent,
        blurEvent,
        boundaryEvent,
        bufferedamountlowEvent,
        bufferedchangeEvent,
        cachedEvent,
        cancelEvent,
        canplayEvent,
        canplaythroughEvent,
        changeEvent,
        chargingchangeEvent,
        chargingtimechangeEvent,
        checkingEvent,
        clickEvent,
        closeEvent,
        closingEvent,
        completeEvent,
        compositionendEvent,
        compositionstartEvent,
        compositionupdateEvent,
        configurationchangeEvent,
        connectEvent,
        connectingEvent,
        connectionstatechangeEvent,
        contentvisibilityautostatechangeEvent,
        contextmenuEvent,
        controllerchangeEvent,
        cookiechangeEvent,
        coordinatorstatechangeEvent,
        copyEvent,
#if ENABLE(APPLE_PAY_COUPON_CODE)
        couponcodechangedEvent,
#endif
        cuechangeEvent,
        currententrychangeEvent,
        cutEvent,
        dataavailableEvent,
        datachannelEvent,
        dblclickEvent,
        dequeueEvent,
        devicechangeEvent,
        devicemotionEvent,
        deviceorientationEvent,
        dischargingtimechangeEvent,
        disconnectEvent,
        disposeEvent,
        downloadingEvent,
        dragEvent,
        dragendEvent,
        dragenterEvent,
        dragleaveEvent,
        dragoverEvent,
        dragstartEvent,
        dropEvent,
        durationchangeEvent,
        emptiedEvent,
        encryptedEvent,
        endEvent,
        endEventEvent,
        endedEvent,
        endstreamingEvent,
        enterEvent,
        enterpictureinpictureEvent,
        errorEvent,
        exitEvent,
        fetchEvent,
        finishEvent,
        focusEvent,
        focusinEvent,
        focusoutEvent,
        formdataEvent,
        fullscreenchangeEvent,
        fullscreenerrorEvent,
        gamepadconnectedEvent,
        gamepaddisconnectedEvent,
        gatheringstatechangeEvent,
        gesturechangeEvent,
        gestureendEvent,
        gesturescrollendEvent,
        gesturescrollstartEvent,
        gesturescrollupdateEvent,
        gesturestartEvent,
        gesturetapEvent,
        gesturetapdownEvent,
        gotpointercaptureEvent,
        hashchangeEvent,
        icecandidateEvent,
        icecandidateerrorEvent,
        iceconnectionstatechangeEvent,
        icegatheringstatechangeEvent,
        inactiveEvent,
        inputEvent,
        inputsourceschangeEvent,
        installEvent,
        invalidEvent,
        invokeEvent,
        keydownEvent,
        keypressEvent,
        keystatuseschangeEvent,
        keyupEvent,
        languagechangeEvent,
        leavepictureinpictureEvent,
        levelchangeEvent,
        loadEvent,
        loadeddataEvent,
        loadedmetadataEvent,
        loadendEvent,
        loadingEvent,
        loadingdoneEvent,
        loadingerrorEvent,
        loadstartEvent,
        lostpointercaptureEvent,
        markEvent,
        merchantvalidationEvent,
        messageEvent,
        messageerrorEvent,
        mousedownEvent,
        mouseenterEvent,
        mouseleaveEvent,
        mousemoveEvent,
        mouseoutEvent,
        mouseoverEvent,
        mouseupEvent,
        mousewheelEvent,
        muteEvent,
        navigateEvent,
        navigateerrorEvent,
        navigatesuccessEvent,
        negotiationneededEvent,
        nexttrackEvent,
        nomatchEvent,
        notificationclickEvent,
        notificationcloseEvent,
        noupdateEvent,
        obsoleteEvent,
        offlineEvent,
        onlineEvent,
        openEvent,
        orientationchangeEvent,
        overflowchangedEvent,
        pagehideEvent,
        pageshowEvent,
        pasteEvent,
        pauseEvent,
        payerdetailchangeEvent,
        paymentauthorizedEvent,
        paymentmethodchangeEvent,
        paymentmethodselectedEvent,
        playEvent,
        playingEvent,
        pointercancelEvent,
        pointerdownEvent,
        pointerenterEvent,
        pointerleaveEvent,
        pointerlockchangeEvent,
        pointerlockerrorEvent,
        pointermoveEvent,
        pointeroutEvent,
        pointeroverEvent,
        pointerupEvent,
        popstateEvent,
        previoustrackEvent,
        processorerrorEvent,
        progressEvent,
        pushEvent,
        pushnotificationEvent,
        pushsubscriptionchangeEvent,
        qualitychangeEvent,
        ratechangeEvent,
        readystatechangeEvent,
        rejectionhandledEvent,
        releaseEvent,
        removeEvent,
        removesourcebufferEvent,
        removestreamEvent,
        removetrackEvent,
        resetEvent,
        resizeEvent,
        resourcetimingbufferfullEvent,
        resultEvent,
        resumeEvent,
        rtctransformEvent,
        scrollEvent,
        searchEvent,
        securitypolicyviolationEvent,
        seekedEvent,
        seekingEvent,
        selectEvent,
        selectedcandidatepairchangeEvent,
        selectendEvent,
        selectionchangeEvent,
        selectstartEvent,
        shippingaddresschangeEvent,
        shippingcontactselectedEvent,
        shippingmethodselectedEvent,
        shippingoptionchangeEvent,
        showEvent,
        signalingstatechangeEvent,
        slotchangeEvent,
        soundendEvent,
        soundstartEvent,
        sourcecloseEvent,
        sourceendedEvent,
        sourceopenEvent,
        speechendEvent,
        speechstartEvent,
        squeezeEvent,
        squeezeendEvent,
        squeezestartEvent,
        stalledEvent,
        startEvent,
        startedEvent,
        startstreamingEvent,
        statechangeEvent,
        stopEvent,
        storageEvent,
        submitEvent,
        successEvent,
        suspendEvent,
        textInputEvent,
        timeoutEvent,
        timeupdateEvent,
        toggleEvent,
        tonechangeEvent,
        touchcancelEvent,
        touchendEvent,
        touchforcechangeEvent,
        touchmoveEvent,
        touchstartEvent,
        trackEvent,
        transitioncancelEvent,
        transitionendEvent,
        transitionrunEvent,
        transitionstartEvent,
        uncapturederrorEvent,
        unhandledrejectionEvent,
        unloadEvent,
        unmuteEvent,
        updateEvent,
        updateendEvent,
        updatefoundEvent,
        updatereadyEvent,
        updatestartEvent,
        upgradeneededEvent,
        validatemerchantEvent,
        versionchangeEvent,
        visibilitychangeEvent,
        voiceschangedEvent,
        volumechangeEvent,
        waitingEvent,
        waitingforkeyEvent,
        webglcontextcreationerrorEvent,
        webglcontextlostEvent,
        webglcontextrestoredEvent,
        webkitAnimationEndEvent,
        webkitAnimationIterationEvent,
        webkitAnimationStartEvent,
        webkitBeforeTextInsertedEvent,
        webkitTransitionEndEvent,
        webkitbeginfullscreenEvent,
        webkitcurrentplaybacktargetiswirelesschangedEvent,
        webkitendfullscreenEvent,
        webkitfullscreenchangeEvent,
        webkitfullscreenerrorEvent,
        webkitkeyaddedEvent,
        webkitkeyerrorEvent,
        webkitkeymessageEvent,
        webkitmouseforcechangedEvent,
        webkitmouseforcedownEvent,
        webkitmouseforceupEvent,
        webkitmouseforcewillbeginEvent,
        webkitneedkeyEvent,
        webkitnetworkinfochangeEvent,
        webkitplaybacktargetavailabilitychangedEvent,
        webkitpresentationmodechangedEvent,
        webkitremovesourcebufferEvent,
        webkitsourcecloseEvent,
        webkitsourceendedEvent,
        webkitsourceopenEvent,
        wheelEvent,
        writeEvent,
        writeendEvent,
        writestartEvent,
        zoomEvent,
    } };
}

} // namespace WebCore
