###############################################################################
##
##  Davtron M811 Digital Clock
##
##  Original: Davtron 803
##  Original from Nasal for DR400-dauphin by dany93;
##                    Clément de l'Hamaide - PAF Team - http://equipe-flightgear.forumactif.com
##  Heavily modified in 2017 for c182s by Benedikt Hallinger
##  Modified in 2020 for M811 by Bea Wolf
##	Reference: https://www.davtron.com/cmsAdmin/uploads/M811_Brochure_012412-.pdf
##  
##  This file is licensed under the GPL license version 2 or later.
##
###############################################################################

# Properties
var volts  = props.globals.getNode("/systems/electrical/DC/lessential-bus/volts", 1);
if( volts.getDoubleValue() == nil ){
	volts.setDoubleValue( 0.0 );
}

var clock    = props.globals.getNode("/instrumentation/clock", 1);
var time_top = clock.getNode("indicated-short-string", 1);
var time_sec = clock.getNode("indicated-sec", 1);
time_sec.setDoubleValue( 0.0 );
var time_ofs = clock.getNode("offset-sec", 1);
var serv     = clock.getNode("serviceable", 1);

print( time_sec.getDoubleValue() );

var davtron = props.globals.initNode("/instrumentation/davtron811");

#		Mode: -1 = ET (Elapsed Time); 0 = FT (Flight Time); 1 = TIME
var mode_p = davtron.initNode("mode", 1, "INT");

#		Dim/BRT: -1 = 1 HR AHEAD; 0 = DIM; 1 = BRT
var dim_switch = davtron.initNode("dim-switch", 0, "INT");

var brightness = davtron.initNode("brightness", 0.5, "DOUBLE");

#		SET: -1 = UP ( 1 min/sec; after 5 sec 5 min/sec ); 0 = nothing; 1 = DN (sets seconds to 0)
var set_switch = davtron.initNode("set-switch", 0, "INT");

#		ET: -1 = RESET; 0 = STOP; 1 = RUN
var et_switch = davtron.initNode("et-switch", 0, "INT");

var et_top = davtron.initNode("elapsed-time", "00:00", "STRING");
var et_sec = davtron.initNode("elapsed-time-secs", 0, "INT");

var ft_top = davtron.initNode("flight-time", "00:00", "STRING");
var ft_sec = davtron.initNode("flight-time-secs", 0, "INT");


############################
# Canvas implementation
############################
# two lines:	upper shows " Hour : Minute "
#		lower shows:  " Second "

var davtron_lcd_test_top    = "88:88";
var davtron_lcd_test_bottom = "88";
var my_canvas = canvas.new({
	"name": "Davtron803",   # The name is optional but allow for easier identification
	"size": [1024, 1024], # Size of the underlying texture (should be a power of 2, required) [Resolution]
	"view": [1024, 1024],  # Virtual resolution (Defines the coordinate system of the canvas [Dimensions]
	# which will be stretched the size of the texture, required)
	"mipmapping": 1       # Enable mipmapping (optional)
});

# add the canvas to replace basic texture of models lcd element
my_canvas.addPlacement({"node": "m811.display"});
#my_canvas.setColorBackground(0, 1, 0, .5);

# create groups holding some stuff
var bggroup  = my_canvas.createGroup();
var lcdgroup = my_canvas.createGroup();

# load davtron png image as background
bggroup.createChild("image")
.setFile("Models/Cockpit/Instruments/M811/M811.png")
.setSize(1024,1024)
.setTranslation(0, 0);      

# Create text element for upper line
var lcdfont = "DSEG/DSEG7/Classic/DSEG7Classic-Bold.ttf";
#var lcdfont = "DSEG/DSEG7/Classic-MINI/DSEG7ClassicMini-Bold.ttf";
var lcd_line_top = lcdgroup.createChild("text", "davtron.top.lcd.text_canvas")
.setTranslation(512, 390)    # The origin is in the top left corner
.setAlignment("center-center") # All values from osgText are supported (see $FG_ROOT/Docs/README.osgtext)
.setFont(lcdfont)            # Fonts are loaded either from $AIRCRAFT_DIR/Fonts or $FG_ROOT/Fonts
.setFontSize(160, 1.0)       # Set fontsize and optionally character aspect ratio
.setColor(0.88,1.0,0)        # Text color
.setText(davtron_lcd_test_top);

# Create text element for lower line
var lcd_line_bot = lcdgroup.createChild("text", "davtron.bot.lcd.text_canvas")
.setTranslation(512, 785)    # The origin is in the top left corner
.setAlignment("center-center") # All values from osgText are supported (see $FG_ROOT/Docs/README.osgtext)
.setFont(lcdfont)            # Fonts are loaded either from $AIRCRAFT_DIR/Fonts or $FG_ROOT/Fonts
.setFontSize(160, 1.0)       # Set fontsize and optionally character aspect ratio
.setColor(0.88,1.0,0)        # Text color
.setText(davtron_lcd_test_bottom);

# Shut dispaly off
var davtron_off = func() {
	lcd_line_top.hide();
	lcd_line_bot.hide();
};

# Turn display on
var davtron_on = func() {
	lcd_line_top.setText(davtron_lcd_test_top);
	lcd_line_bot.setText(davtron_lcd_test_bottom);
	lcd_line_top.show();
	lcd_line_bot.show();
};

# Update the state of the selected mode
var davtron_updateMode = func(){
	var value_top = davtron_lcd_test_top;
	var value_btm = davtron_lcd_test_bottom;
	var mode = mode_p.getIntValue();
	if( mode == 1 ){
		value_top = time_top.getValue();
		value_btm = sprintf("%02d", math.mod(time_sec.getDoubleValue(), 60) );
	} else if( mode == 0 ){
		value_top = ft_top.getValue();
		value_btm = sprintf("%02d", math.mod(ft_sec.getDoubleValue(), 60) );
	} else if( mode == -1 ){
		value_top = et_top.getValue();
		value_btm = sprintf("%02d", math.mod(et_sec.getDoubleValue(), 60) );
	} else {
		print("Davtron 811: Unknown mode!");
	}
	lcd_line_top.setText(value_top);
	lcd_line_bot.setText(value_btm);
};

# Calculate current state and update clock
var davtron_on_off = func() {
	# see if we have power and are serviceable
	if ( serv.getBoolValue() and volts.getDoubleValue() > 9) {
		if (lcd_line_top.getVisible() != 1) davtron_on();
	} else {
		# shut down
		davtron_off();
	}
};




##########################################
# Flight timer / Elapsed timer
# Note: Currently only counting upwards is supported; whilst the device also supports count down mode!
##########################################


# Floor function
var floor = func(v) v < 0.0 ? -int(-v) - 1 : int(v);

# Time format function: Format time of property from secs to HH:MM format
var timeFormat = func(timeProp){
	
	elapsedTime = getprop(timeProp);
	
	hrs = floor(elapsedTime/3600);
	min = floor(elapsedTime/60);
	
	formattedTime = sprintf("%02d:%02d", hrs, min-(60*hrs));
	
	return formattedTime;
}

setlistener(et_switch, func( i ){
	if( mode_p.getIntValue() == -1 ){
		if( i.getIntValue() == 1 ){
			davtron_elapsed_time.start();
		} else if ( i.getIntValue() == 0 ){
			davtron_elapsed_time.stop();
		} else if ( i.getIntValue() == -1 ){
			davtron_elapsed_time.reset();
		} else {
			print("Davtron 811: Unknown ET switch position!");
		}
	}
});

setlistener(dim_switch, func( i ){
	if( i.getIntValue() == -1 ){
		# set time one hour ahead
		var val = time_ofs.getValue();
		if( val >= 82800 ){
			val = 0;
		} else {
			val += 3600;
		}
		time_ofs.setValue( val );
	} else if ( i.getIntValue() == 0 ){
		brightness.setDoubleValue( 0.5 );
	} else if ( i.getIntValue() == 1 ){
		brightness.setDoubleValue( 1.0 );
	} else {
		print("Davtron 811: Unknown DIM switch position!");
	}
});

var dt_p = props.globals.getNode("sim/time/delta-sec");

var held_loop = func {
	var dt = dt_p.getDoubleValue();
	if( held == 0 ) {
		held_loop_timer.stop();
		return;
	}
	if( seconds_held > 5.0 ) {
		time_ofs.setValue( time_ofs.getValue() + 300 * dt );
	} else {
		time_ofs.setValue( time_ofs.getValue() + 60 * dt );
	}
	seconds_held += dt;
}

var held_loop_timer = maketimer( 0.0, held_loop );

var held = 0;
var seconds_held = 0.0;

setlistener(set_switch, func( i ){
	if( i.getIntValue() == -1 ){
		# Time up: Increase 1 minute per second; after 5 seconds increase 5 minutes per second
		held = 1;
		held_loop_timer.start();
	} else if( i.getIntValue() == 0 ){
		#nothing
		held = 0;
	} else if( i.getIntValue() == 1 ){
		# Time down: Set seconds to 0
		held = 0;
		time_ofs.setValue( time_ofs.getValue() - math.mod( time_sec.getDoubleValue(), 60 ) );
	} else {
		print("Davtron 811: Unknown SET switch position!");
	}
});

###########
# INIT
###########
#print("Davtron 803 init: state");

# init state
davtron_on_off();
controlBtnPressedAt = 0;
elapsedTimeResetMarker = 0;

# init timers (API details: http://api-docs.freeflightsim.org/fgdata/aircraft_8nas_source.html )
props.globals.initNode("/instrumentation/davtron811/flight-time-secs",  0, "INT");
props.globals.initNode("/instrumentation/davtron811/elapsed-time-secs", 0, "INT");
var davtron_flight_time  = aircraft.timer.new("/instrumentation/davtron811/flight-time-secs", 1, 0);
var davtron_elapsed_time = aircraft.timer.new("/instrumentation/davtron811/elapsed-time-secs", 1, 0);

# Activate the FT timer at startup of elec system
# and stop if no power registered (note, the ET timer seems to count on as seen in real life. FT timer behavior is not verified in this way and may be wrong)
setlistener(volts, func(clocknode) {
	if (clocknode.getValue() > 1) {
		davtron_flight_time.start();
	} else {
		davtron_flight_time.stop();
	}
}, 1, 0);

# Generate formatted output in separate properties
timeFormatUpdateLoop = maketimer(0.1, func(){
	setprop("/instrumentation/davtron811/flight-time", timeFormat("/instrumentation/davtron811/flight-time-secs"));
	setprop("/instrumentation/davtron811/elapsed-time", timeFormat("/instrumentation/davtron811/elapsed-time-secs"));
	
	# Update the LCD values
	davtron_updateMode();
});
timeFormatUpdateLoop.start();

# Listen to changes in power and serviceable to enable/disable the clock
setlistener(serv, davtron_on_off, 1, 0);
setlistener(volts, davtron_on_off, 1, 0);

# Listen to changes in config so davtron responds timely
setlistener("/instrumentation/davtron811/mode", davtron_updateMode, 1, 0);


print("Davtron M811 initialized");
