##########################################################
####### Single Starter/Generator electrical system #######
####################    Syd Adams    #####################
###### Based on Curtis Olson's nasal electrical code #####
##########################################################
##########################################################
######## Modified by Clement DE L'HAMAIDE for P92 ########
##########################################################
# Modified by Benedikt Wolf for P130UL (2021-2023)

# Based on information from http://www.aircraft-battery.com/search-by-your-aircraft/battery_detail/156

var OutPuts = props.globals.getNode("/systems/electrical/outputs",1);
var Volts = props.globals.getNode("/systems/electrical/volts",1);
var Amps = props.globals.getNode("/systems/electrical/amps",1);
var Battery={};
var Alternator={};
var load = 0.0;

var delta_t = props.globals.getNode("/sim/time/delta-sec");

var sw = props.globals.getNode("/controls/switches");
var sw_e = props.globals.getNode("/controls/electric");
var sw_l = props.globals.getNode("/controls/lighting");
var cb = props.globals.initNode("/controls/circuit-breakers");

var serviceable = {
	electric:	props.globals.getNode("systems/electrical/serviceable",1),
	comm:		props.globals.getNode("instrumentation/comm/serviceable"),
};
var switches = {
	bat:			props.globals.initNode("/controls/engines/engine/master-bat", 0, "BOOL"),
	alt:			props.globals.initNode("/controls/engines/engine/master-alt", 0, "BOOL"),
	radio_master:	sw_e.initNode("radio-master",	1,	"BOOL"),
	starter:		props.globals.getNode("/controls/engines/engine[0]/starter-switch", 1),
	strobe:		sw_l.initNode("strobe",		0,	"BOOL"),
	boost:		props.globals.initNode("/controls/engines/engine/fuel-pump", 0, "BOOL"),
};
var circuit_breakers = {
	radio:	cb.initNode("radio",	1,	"BOOL"),
	flash:	cb.initNode("flash",	1,	"BOOL"),
	fuel_p:	cb.initNode("fuel-pump",1,	"BOOL"),
	alt:		cb.initNode("gen",	1,	"BOOL"),
	batt:		cb.initNode("batt",	1,	"BOOL"),
};
var comm_output = OutPuts.initNode("comm", 0.0, "DOUBLE");
var eng_amp = props.globals.initNode("/engines/engine/amp-v", 0.0, "DOUBLE");

# Estimate electrical load:
# 	Strobe Light based on https://aeroleds.com/wp-content/uploads/2018/02/0008-0006-Linear-Pulsar-NS-Series-Installation.pdf:
#	Strobe:		70 Watts (during flash, 0.2 secs)	-> *2
#				11.2 Watts (average)			-> *2
# 	Fuel Pump: 		50 Watts (complete guess)
var consumers_main = { 
	# consumer name:  [  power (watts),  switch node,   circuit breaker node,  output node ],
	strobe_light:	[ 22.4, switches.strobe, circuit_breakers.flash,	OutPuts.initNode("strobe-light")	],
	fuel_pump:		[ 50.0, switches.boost, circuit_breakers.fuel_p,	OutPuts.initNode("fuel-pump")		],
	indicators:		[  5.0, nil, nil,	OutPuts.initNode("indicators")		],
	# Starter has about 900W power according to https://www.ulforum.de/ultraleicht/forum/2_technik-und-flugzeuge/2466_anlasser-rotax-912-uls/seite-4.html
	starter:		[ 900.0, switches.starter, nil, OutPuts.initNode("starter") ],
};

var strobe = aircraft.light.new("/sim/model/lights/strobe-light", [0.2, 1.25], "/systems/electrical/outputs/strobe-light");

#var battery = Battery.new(volts,amps,amp_hours,charge_norm,charge_amps);

Battery = {
	new : func( ideal_volts, ideal_amps, amp_hours, charge_norm, charge_amps ) {
		m = { parents : [Battery] };
		m.ideal_volts = ideal_volts;
		m.ideal_amps = ideal_amps;
		m.amp_hours = amp_hours;
		m.charge_norm = charge_norm;
		m.charge_amps = charge_amps;
		return m;
	},
	apply_load : func {
		var amphrs_used = arg[0] * (arg[1] / 3600.0);
		var percent_used = amphrs_used / me.amp_hours;
		me.charge_norm -= percent_used;
		if ( me.charge_norm < 0.0 ) {
			me.charge_norm = 0.0;
		} elsif ( me.charge_norm > 1.0 ) {
			me.charge_norm = 1.0;
		}
		return me.amp_hours * me.charge_norm;
	},
	get_output_volts : func {
		var x = 1.0 - me.charge_norm;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_volts * factor;
	},
	get_output_amps : func {
		var x = 1.0 - me.charge_norm;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_amps * factor;
	},	
	recharge : func {
		me.charge_norm = 1.0;
	},
};

# var alternator = Alternator.new("rpm-source",rpm_threshold,volts,amps);

Alternator = {
	new : func( rpm_source, rpm_threshold, ideal_volts, ideal_amps ) {
		m = { parents : [Alternator] };
		m.rpm_source =  props.globals.getNode( rpm_source,1);
		m.rpm_threshold = rpm_threshold;
		m.ideal_volts = ideal_volts;
		m.ideal_amps = ideal_amps;
		return m;
	},
	
	apply_load : func( amps, dt) {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ){
			factor = 1.0;
		}
		var available_amps = me.ideal_amps * factor;
		return available_amps - amps;
	},
	
	get_output_volts : func {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ) {
			factor = 1.0;
		}
		return me.ideal_volts * factor;
	},
	
	get_output_amps : func {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ) {
			factor = 1.0;
		}
		return me.ideal_amps * factor;
	}
};

var battery = Battery.new(12.0, 20.0, 22.0, 1.0, 12.0);
var alternator = Alternator.new("/engines/engine[0]/rpm", 250.0, 14.0, 40.0);

var update_virtual_bus = func() {
	var bus_volts = 0.0;
	var dt = delta_t.getDoubleValue();
	var AltVolts = alternator.get_output_volts();
	var AltAmps = alternator.get_output_amps();
	var BatVolts = battery.get_output_volts();
	var BatAmps = battery.get_output_amps();
	var power_source = nil;
	
	if( serviceable.electric.getBoolValue() ){					# Electrical System is serviceable
		if ( switches.alt.getBoolValue() and circuit_breakers.alt.getBoolValue() and (AltVolts > BatVolts)){
			bus_volts = AltVolts;
			power_source = "alternator";
			battery.apply_load(-battery.charge_amps, dt);			# Charge the battery
		}elsif ( switches.bat.getBoolValue() and circuit_breakers.batt.getBoolValue() ){
			bus_volts = BatVolts;
			power_source = "battery";
			battery.apply_load(load, dt);	# Load in ampere
		}else {
			bus_volts = 0.0;
		}
	} else {									# Electrical System not serviceable
		bus_volts = 0.0;
	}
	
	load = 0.0;
	load += electrical_bus(bus_volts);
	load += avionics_bus(bus_volts);
	
	eng_amp.setDoubleValue(bus_volts);		# TODO what does this affect?
	
	var bus_amps = 0.0;
	
	if (bus_volts > 1.0){
		if (power_source == "battery"){
			bus_amps = BatAmps-load;
		} else {
			bus_amps = battery.charge_amps;
		}
	}
	
	Amps.setDoubleValue(bus_amps);
	Volts.setDoubleValue(bus_volts);
	return load;
}

var electrical_bus = func(bus_volts){
	load = 0.0;
	
	foreach( var key; keys( consumers_main ) ){
		if( ( consumers_main[key][1] == nil or consumers_main[key][1].getBoolValue() ) and ( consumers_main[key][2] == nil or consumers_main[key][2].getBoolValue() ) ){
			consumers_main[key][3].setDoubleValue( bus_volts );
			if( bus_volts != 0 ){
				load += ( consumers_main[key][0] / bus_volts );
			}
		} else {
			consumers_main[key][3].setDoubleValue( 0.0 );
		}
	}
	
	return load;
}

var avionics_bus = func(bv) {
	load = 0.0;
	var bus_volts = 0.0;
	
	if( switches.radio_master.getBoolValue() ) {
		bus_volts = bv;
	}
		
	if ( serviceable.comm.getBoolValue() and circuit_breakers.radio.getBoolValue() and bus_volts > 0.0 ){
		comm_output.setDoubleValue( bus_volts );
		load += 0.02 / bus_volts;
	} else {
		comm_output.setDoubleValue( 0.0 );
	}
	
	return load;
}

var electrical_updater = maketimer( 0.0, update_virtual_bus );
electrical_updater.simulatedTime = 1;

setlistener("/sim/signals/fdm-initialized", func {
	electrical_updater.start();
});
