##
# Procedural model of a Alexander Schleicher Ka6 electrical system.
# Based on C172P electrical system.

# Initialize properties
var com_ptt = props.globals.getNode("/instrumentation/comm[0]/ptt", 1);
var com_start = props.globals.getNode("/instrumentation/comm[0]/start", 1);
var vario_vol = props.globals.getNode("/instrumentation/ilec-sc7/volume", 1);
var vario_aud = props.globals.getNode("/instrumentation/ilec-sc7/audio", 1);
var vario_read = props.globals.getNode("/instrumentation/ilec-sc7/te-reading-mps", 1);
var turnbank_spin = props.globals.getNode("/instrumentation/turn-indicator/spin", 1);

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

var electrical = props.globals.getNode("/systems/electrical", 1);

var output = electrical.getNode("outputs", 1);
var cb = props.globals.getNode("/controls/circuit-breakers/", 1);
var sw = props.globals.getNode("/controls/electric/", 1);
var batt_prop = electrical.getNode( "battery", 1 );

var serviceable = electrical.initNode( "serviceable", 1 , "BOOL" );
var amps  = electrical.initNode( "amps",  0.0, "DOUBLE");
var volts = electrical.initNode( "volts", 0.0, "DOUBLE");

var batt_connector = props.globals.initNode("/controls/electric/battery-connector", 1, "BOOL");

var breakers = {
	master:	cb.initNode( "master", 	1, "BOOL"),
	com:	cb.initNode( "radio", 	1, "BOOL"),
	vario:	cb.initNode( "vario", 	1, "BOOL"),
	turn:	cb.initNode( "turn",	1, "BOOL"),
};

var outputs = {
	com:	output.initNode( "comm", 		0.0, "DOUBLE"),
	flarm:	output.initNode( "flarm", 		0.0, "DOUBLE"),
	vario:	output.initNode( "ilec-sc7", 		0.0, "DOUBLE"),
	turn:	output.initNode( "turn-coordinator",	0.0, "DOUBLE"),
};

var switches = {
	turn:		sw.initNode( "turn-switch", 0, "BOOL" ),
	flarm:	props.globals.getNode( "/instrumentation/FLARM/master-switch", 1 ),
};

##
# Initialize internal values
#

var ammeter_ave = 0.0;

##
# Battery model class.
#

var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, charge_amps, n ){
		var charge_prop	= batt_prop.getNode( "charge["~n~"]" );
		var charge	= nil;
		if( getprop("/systems/electrical/battery/charge["~n~"]") != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = batt_prop.initNode("charge["~n~"]", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			ideal_volts: ideal_volts,
			ideal_amps: ideal_amps,
			amp_hours: amp_hours,
			charge_amps: charge_amps,
			charge: charge,
			charge_prop: charge_prop,
			n: n 
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps, dt ){
		var old_charge = me.charge_prop.getDoubleValue();
		if( freeze_replay.getBoolValue() ){
			return me.amp_hours * old_charge;
		}
		var amphrs_used = amps * dt / 3600.0;
		var fraction_used = amphrs_used / me.amp_hours;
		
		var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));
		
		if (new_charge < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
		me.charge = new_charge;
		me.charge_prop.setDoubleValue( new_charge );
		return me.amp_hours * new_charge;
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32 ) / 32;
		return me.ideal_volts * factor;
	},
	# Return output amps available.  This function is totally wrong and should be
	# fixed at some point with a more sensible function based on charge percent.
	# There is probably some physical limits to the number of instantaneous amps
	# a battery can produce (cold cranking amps?)
	get_output_amps: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32) / 32;
		return me.ideal_amps * factor;
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};


var battery = BatteryClass.new( 12.0, 9.0, 9.0, 7.0, 0);

var reset_circuit_breakers = func {
	# Reset circuit breakers
	foreach( var el; breakers ){
		el.setBoolValue( 1 );
	}
}

var recharge_battery = func {
	# Charge battery to 100 %
	battery.reset_to_full_charge();
}
##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_virtual_bus(dt);
	}
};

##
# Model the system of relays and connections that join the battery,master/alt switches.
#

var update_virtual_bus = func (dt) {
	
	var load = 0.0;
	var battery_volts = 0.0;
	if ( serviceable.getBoolValue() ) {
		battery_volts = battery.get_output_volts();
	}
	
	
	# determine power source
	if ( batt_connector.getBoolValue() ){
		var bus_volts = battery_volts;
		var power_source = battery;
	}else{
		var bus_volts = 0.0;
	}
	#print( "virtual bus volts = ", bus_volts );
	
	# bus network (1. these must be called in the right order, 2. the
	# bus routine itself determins where it draws power from.)
	load += electrical_bus_1( bus_volts );
	
	# swtich the master breaker off if load is out of limits
	if ( load > 55 ) {
		bus_volts = 0;
	}
	
	# system loads and ammeter gauge
	var ammeter = 0.0;
	if ( bus_volts > 1.0 ) {
		# ammeter gauge
		if ( power_source == "battery" ) {
			ammeter = -load;
		} else {
			ammeter = battery.charge_amps;
		}
	}
	# print( "ammeter = ", ammeter );
	
	# charge/discharge the battery
	battery.apply_load( load, dt );
	
	# filter ammeter needle pos
	ammeter_ave = 0.8 * ammeter_ave + 0.2 * ammeter;
	
	# outputs
	amps.setDoubleValue( ammeter_ave );
	volts.setDoubleValue( bus_volts );
}

#Load sources:
#	com:		https://www.skyfox.com/becker-ar6201-022-vhf-am-sprechfunkgeraet-8-33.html
#	vario:		http://www.ilec-gmbh.com/ilec/manuals/SC7pd.pdf
#	turn:		https://www.airteam.eu/de/p/falcon-tb02e-2-1 (not the same but similar)
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
#	flarm display:	https://www.air-store.eu/Display-V3-FLARM

var electrical_bus_1 = func( bus_volts ) {
	var load = 0.0;
	
	# check master breaker
	if ( !breakers.master.getBoolValue() ) {
		# we are feed from the virtual bus
		bus_volts = 0.0;        
	}
	#print("Bus volts: ", bus_volts);
	
	if(bus_volts > 9){
		
		# Vario
		if( breakers.vario.getBoolValue() ){
			outputs.vario.setDoubleValue( bus_volts );
			#Energy consumption:	25mA (medium volume) 60mA (max volume) -> guess: at 12V
			#			guess: base consumption 5mA (no volume)
			load += 0.06 / bus_volts;
			if(vario_aud.getValue() == 2 or (vario_aud.getValue() == 1 and vario_read.getValue() > 0)){
				load += (vario_vol.getValue()*0.66) / bus_volts;
			}
		}else{
			outputs.vario.setDoubleValue( 0.0 );
		}
		
		
		# Radio
		if( breakers.com.getBoolValue() ){
			outputs.com.setDoubleValue( bus_volts );
			if(com_ptt.getBoolValue() and com_start.getValue()==1){
				load += 19.2 / bus_volts;
			}else{
				load += 1.02*com_start.getValue() / bus_volts;
			}
		}else{
			outputs.com.setDoubleValue( 0.0 );
		}
		
		#Turn Coordinator
		#Energy Consumption:
		#	starting ~9.9 / volts (approx)
		#	running ~7.8 / volts (approx)
		if( breakers.turn.getBoolValue() and switches.turn.getBoolValue() ) {
			outputs.turn.setDoubleValue( bus_volts );
			if( turnbank_spin.getValue() > 0.99 ){
				load += 7.8 / bus_volts;
			}else{
				load += 9.9 / bus_volts;
			}
		} else {
			outputs.turn.setDoubleValue( 0.0 );
		}
		
		if( switches.flarm.getBoolValue() ){
			outputs.flarm.setDoubleValue( bus_volts );
			load += 0.66 / bus_volts; #FLARM
			load += 0.12 / bus_volts; #FLARM display
		}else{
			outputs.flarm.setDoubleValue( 0.0 );
		}
	}else{
		outputs.vario.setDoubleValue( 0.0 );
		outputs.com.setDoubleValue( 0.0 );
		outputs.turn.setDoubleValue( 0.0 );
		outputs.flarm.setDoubleValue( 0.0 );
	}
	
	# return cumulative load
	return load;
}

##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
	# checking if battery should be automatically recharged
	if (!getprop("/systems/electrical/save-battery-charge")) {
		battery.reset_to_full_charge();
	};
	
	system_updater.enable();
	print("Electrical system initialized");
});
