##
# Pipistrel Taurus Electro G2.5 Electrical System
#	by Benedikt Wolf (D-ECHO) 05/2020
#	Reference:	[1]	p.67 - p.72

#	adapted from: Cessna 172p Electrical System


#Motor System: 
# max voltage: 118V, min voltage: 90V

##
# Buses:
# Avionics Bus (DC)
# Engine Bus (AC)
#

var volts = 0.0;

var bus_volts = 0.0;

var ammeter_ave = 0.0;

# Initialize properties
var electrical	=	props.globals.initNode("/systems/electrical");
var serviceable	=	electrical.initNode("serviceable", 1, "BOOL");
var batt_prop	=	electrical.initNode("battery");
var bus_prop	=	electrical.initNode("bus");
var output_prop	=	electrical.initNode("outputs");
var cb_prop	=	props.globals.initNode("/controls/circuit-breakers");

var avrg_chrg	=	electrical.initNode("average-charge", 0.0, "DOUBLE");

var engine_amps		=	bus_prop.initNode("engine-amps", 0.0, "DOUBLE");
var engine_volts	=	bus_prop.initNode("engine-volts", 0.0, "DOUBLE");

var cockpit_amps	=	bus_prop.initNode("cockpit-amps", 0.0, "DOUBLE");
var cockpit_volts	=	bus_prop.initNode("cockpit-volts", 0.0, "DOUBLE");

var prop_cmd		=	props.globals.getNode("/instrumentation/esys-man/position-cmd");
var prop_cmd_int	=	props.globals.getNode("/fdm/jsbsim/propulsion/engine[0]/propeller-tower-cmd-norm");

var vario	=	output_prop.initNode("S3", 0.0, "DOUBLE");
var radio	=	output_prop.initNode("comm[0]", 0.0, "DOUBLE");
var esys	=	output_prop.initNode("esys", 0.0, "DOUBLE");
var transponder	=	output_prop.initNode("transponder", 0.0, "DOUBLE");
var flarm	=	output_prop.initNode("flarm", 0.0, "DOUBLE");

var xpdr_mode	=	props.globals.getNode("/instrumentation/transponder[0]/inputs/knob-mode");
var comm_ptt	=	props.globals.getNode("/instrumentation/comm[0]/ptt");
var esys_master	=	props.globals.initNode("/instrumentation/esys-man/master", 0, "BOOL");

# Array of circuit breakers
var circuit_breakers = {
	master: cb_prop.initNode("master", 1, "BOOL"),
	cockpit: cb_prop.initNode("cockpit", 1, "BOOL"),
};

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

##
# Battery model class.
#

var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, charge_amps, n ){
		var charge_prop	= batt_prop.getNode( "charge["~n~"]" );
		var charge	= nil;
		if( charge_prop.getDoubleValue() != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = batt_prop.initNode("charge["~n~"]", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			ideal_volts: ideal_volts,
			ideal_amps: ideal_amps,
			amp_hours: amp_hours,
			charge_amps: charge_amps,
			charge: charge,
			charge_prop: charge_prop,
			n: n 
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps, dt ){
		var old_charge = me.charge_prop.getDoubleValue();
		if( freeze_replay.getBoolValue() ){
			return me.amp_hours * old_charge;
		}
		var amphrs_used = amps * dt / 3600.0;
		var fraction_used = amphrs_used / me.amp_hours;
		
		var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));
		
		if (new_charge < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
		me.charge = new_charge;
		me.charge_prop.setDoubleValue( new_charge );
		return me.amp_hours * new_charge;
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32 ) / 32;
		return me.ideal_volts * factor;
	},
	# Return output amps available.  This function is totally wrong and should be
	# fixed at some point with a more sensible function based on charge percent.
	# There is probably some physical limits to the number of instantaneous amps
	# a battery can produce (cold cranking amps?)
	get_output_amps: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32) / 32;
		return me.ideal_amps * factor;
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};

############################
####	Battery Packs	####
############################

##	The Taurus Electro has 4 battery boxes, each with the following specifications:
##		Voltage:	max:	270V
##				min:	190V
##
##		Capacity:	1.1875 kWh	=	1187.5 Wh	(0.95 kWh recommended usable)
##					From P = U * I:
##					I = P / U
##					I * t = (P * t) / U
##				assume:	U = 230V
###
##		Amp hours:	5.163 Ah
##
##		Max. Amperage:	200 A (breaker)
##			assume: ideal amperage: 180 A

##	Batteries are organised in an array:
var ideal_volts = 230;
var ideal_amps = 180; 
var amp_hours = 5.163;
var charge_amps = 4;

var batteries = [
	BatteryClass.new( ideal_volts, ideal_amps, amp_hours, charge_amps, 0),
	BatteryClass.new( ideal_volts, ideal_amps, amp_hours, charge_amps, 1),
	BatteryClass.new( ideal_volts, ideal_amps, amp_hours, charge_amps, 2),
	BatteryClass.new( ideal_volts, ideal_amps, amp_hours, charge_amps, 3),
];

var reset_battery = func {
	# Charge batteries to 100 %
	foreach(var batt; batteries){
		batt.reset_to_full_charge();
	}
}
var reset_circuit_breakers = func {
	# Reset circuit breakers
	foreach(var cb; keys(circuit_breakers)){
		circuit_breakers[cb].setBoolValue( 1 );
	}
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_bus(dt);
	}
};

var voltages = [0, 0, 0, 0];

var update_bus = func (dt) {
	if( !serviceable.getBoolValue() ){
		return;
	}
	#var external_volts = 0.0;
	var load = 0.0;
	
	var n = 0;
	var bus_volts = 0;
	forindex( var i; batteries){
		voltages[i] = batteries[i].get_output_volts();
		if( voltages[i] > 190 ){
			n = n + 1;
			bus_volts = bus_volts + voltages[i];
		}
	}
	bus_volts = bus_volts / n;
	
	# switch state
	var master_switch = getprop("/systems/FES/fes-master-switch");
	load += cockpit_bus( bus_volts );
	load += engine_bus( bus_volts );
	
	if ( load > 200 ) {
		circuit_breakers.master.setBoolValue( 0 );
	}
	
	var sum_charge = 0;
	# charge/discharge the batteries
	forindex( var i; batteries ){
		if( voltages[i] > 190 ){
			batteries[i].apply_load( load / n, dt );
		}
		sum_charge = sum_charge + batteries[i].get_charge();
	}
	avrg_chrg.setDoubleValue( sum_charge / 4 );
}

var cockpit_bus = func( bv ) {
	var load = 0.0;
	var bus_volts = 0.0;
	if( circuit_breakers.cockpit.getBoolValue() ){
		bus_volts = bv;
	}
	# Convert to low voltage
	if( bus_volts > 190 ){
		bus_volts = bus_volts / 19.3;
	} else {
		bus_volts = 0.0;
	}
	
	# Electrical Consumers (Instruments)
	# Vario
	vario.setDoubleValue( bus_volts );
	load += bus_volts / 177.78; #135mA
	
	
	# Radio
	radio.setDoubleValue( bus_volts );
	if( comm_ptt.getBoolValue() ){
		load += bus_volts / 15; #transmitting: 1600mA
	}else{
		load += bus_volts / 282.353; #85mA (standby)
	}
	
	# ESYS-MAN
	if ( esys_master.getBoolValue() ) {
		esys.setDoubleValue( bus_volts );
		load += bus_volts / 240; #100mA
	} else {
		esys.setDoubleValue( 0.0 );
	}
	
	# Transponder
	transponder.setDoubleValue( bus_volts );
	if(xpdr_mode.getIntValue() > 0 and bus_volts != 0){
		load += 18 / bus_volts;	#1.5A at 12VDC
	}
	
	# FLARM
	flarm.setDoubleValue( bus_volts );
	
	cockpit_amps.setDoubleValue( load );
	cockpit_volts.setDoubleValue( bus_volts );
	
	if( load > 20 ){
		circuit_breakers.cockpit.setBoolValue( 0 );
	}
	
	return load;
}

var engine_bus = func( bv ) {
	var bus_volts = 0.0;
	var load = 0.0;
	if( bv > 0 ){
		bus_volts = bv;
	}
	
	if( bus_volts > 190 ){
		#Load is engine power divided by volts
		var eng_power = ( getprop("fdm/jsbsim/propulsion/engine[0]/power-hp") or 0 ) * 745.7; #hp to watts
		load += eng_power / bus_volts;
		prop_cmd_int.setBoolValue( prop_cmd.getBoolValue() );
	}
	
	engine_amps.setDoubleValue( load );
	engine_volts.setDoubleValue( bus_volts );
	
	return load;
}


##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
	reset_circuit_breakers();
	
	system_updater.enable();
	print("Electrical System initialized");
});

