##
# Front Electric Sustainer System
#

#Motor System: 
# max voltage: 118V, min voltage: 90V

##
# Buses:
# Avionics Bus (DC)
# Engine Bus (AC)
#

var volts = 0.0;

var bus_volts = 0.0;

var ammeter_ave = 0.0;

##
# Battery model class.
#

var BatteryClass = {};

BatteryClass.new = func (volts, amps, amp_hrs, name) {
    var obj = { parents : [BatteryClass],
                ideal_volts : volts,
                ideal_amps : amps,
                amp_hours : amp_hrs,
                name : name,
                charge_percent : getprop("/systems/FES/battery-charge-percent-"~name) or 1.0,
                charge_amps : 7.0 };
    setprop("/systems/FES/battery-charge-percent-"~name, obj.charge_percent);
    return obj;
}

##
# Passing in positive amps means the battery will be discharged.
# Negative amps indicates a battery charge.
#

BatteryClass.apply_load = func (amps, dt) {
    var old_charge_percent = getprop("/systems/FES/battery-charge-percent-"~me.name);

    if (getprop("/sim/freeze/replay-state"))
        return me.amp_hours * old_charge_percent;

    var amphrs_used = amps * dt / 3600.0;
    var percent_used = amphrs_used / me.amp_hours;

    var new_charge_percent = std.max(0.0, std.min(old_charge_percent - percent_used, 1.0));

    if (new_charge_percent < 0.1 and old_charge_percent >= 0.1)
        gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
    me.charge_percent = new_charge_percent;
    setprop("/systems/FES/battery-charge-percent-"~me.name, new_charge_percent);
    return me.amp_hours * new_charge_percent;
}

##
# Return output volts based on percent charged.  Currently based on a simple
# polynomial percent charge vs. volts function.
#

BatteryClass.get_output_volts = func {
    var x = 1.0 - me.charge_percent;
    var tmp = -(3.0 * x - 1.0);
    var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
    return me.ideal_volts * factor;
}


##
# Return output amps available.  This function is totally wrong and should be
# fixed at some point with a more sensible function based on charge percent.
# There is probably some physical limits to the number of instantaneous amps
# a battery can produce (cold cranking amps?)
#

BatteryClass.get_output_amps = func {
    var x = 1.0 - me.charge_percent;
    var tmp = -(3.0 * x - 1.0);
    var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
    return me.ideal_amps * factor;
}

##
# Set the current charge instantly to 100 %.
#

BatteryClass.reset_to_full_charge = func {
    me.apply_load(-(1.0 - me.charge_percent) * me.amp_hours, 3600);
}


var battery_one = BatteryClass.new(118, 200, 301,"one");
var battery_two = BatteryClass.new(118, 200, 301,"two");

var reset_battery_and_circuit_breakers = func {
    # Charge battery to 100 %
    battery_front.reset_to_full_charge();
    battery_back.reset_to_full_charge();

    # Reset circuit breakers
    setprop("/controls/circuit-breakers/fes-master", 1);
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
    new : func {
        var m = {
            parents: [ElectricalSystemUpdater]
        };
        # Request that the update function be called each frame
        m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
        return m;
    },

    enable: func {
        me.loop.reset();
        me.loop.enable();
    },

    disable: func {
        me.loop.disable();
    },

    reset: func {
        # Do nothing
    },

    update: func (dt) {
        update_bus(dt);
    }
};


var update_bus = func (dt) {
    var serviceable = getprop("/systems/FES/serviceable");
    var external_volts = 0.0;
    var load = 0.0;
    var battery_one_volts = 0.0;
    var battery_two_volts = 0.0;
    if ( serviceable ) {
        battery_one_volts = battery_one.get_output_volts();
        battery_two_volts = battery_two.get_output_volts();
    }

    # switch state
    var master_switch = getprop("/systems/FES/fes-master-switch");

    # determine power source
    var bus_volts = 0.0;
    var power_source = nil;
    if ( master_switch ) {
	bus_volts = (battery_one_volts+battery_two_volts)/2;
    }
    
    #setprop("/systems/FES/batt1volts", battery_one_volts);
    #setprop("/systems/FES/batt2volts", battery_two_volts);

    load += engine_bus_1(bus_volts);

    # swtich the master breaker off if load is out of limits
    if ( load > 325 ) {
      setprop("controls/circuit-breakers/fes-master", 0);
    }

    

    # charge/discharge the battery
        battery_one.apply_load( 0.5*load, dt );
        battery_two.apply_load( 0.5*load, dt );


    # outputs
    if(load != nil){
	setprop("/systems/FES/amps", load);
	}
    setprop("/systems/FES/volts", bus_volts);
}

var engine_bus_1 = func(bv) {
    var bus_volts = 0.0;
    var load = 0.0;
    
    bus_volts = bv;
    #print("Eng Bus volts: ", bus_volts);
    
	setprop("systems/FES/engine-output-volts", bus_volts);

	#Load is engine power divided by volts
	var eng_power = getprop("fdm/jsbsim/propulsion/engine[0]/power-hp") * 745.7; #hp to watts
	load += eng_power / bus_volts;

    # register bus voltage
    eng_bus1_volts = bus_volts;

    # return cumulative load
    return load;
}


##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
    # checking if battery should be automatically recharged
        battery_one.reset_to_full_charge();
        battery_two.reset_to_full_charge();
    setprop("/controls/circuit-breakers/fes-master", 1);

    system_updater.enable();
    print("FES System initialized");
});

