/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra.conversation.jsf.components;

import java.io.IOException;

import javax.faces.component.UIComponentBase;
import javax.faces.context.FacesContext;

import org.apache.myfaces.orchestra.conversation.ConversationRequestParameterProvider;
import org.apache.myfaces.shared_orchestra.renderkit.RendererUtils;

/**
 * Embedded links will start a new conversation context.
 * <p>
 * Use this as the ancestor of any clickable links that should cause the new page to
 * run within a totally new set of conversations. This really only makes sense when
 * those links cause a new window to open. That new window then acts completely
 * independently of the original window (as long as data is stored in conversations,
 * not directly in the http session).
 * <p>
 * Normally, all urls within a view are rendered with a query parameter holding the
 * current conversation context id, causing a later request to that url to use the
 * same context as was used to render the original page. This tag causes this special
 * query parameter to be omitted for all urls output by nested components. When any
 * such url is invoked, the missing conversation context id then causes a new
 * conversation context to be created.
 */
public class UISeparateConversationContext extends UIComponentBase
{
    public static final String COMPONENT_FAMILY = "javax.faces.Component";
    public static final String COMPONENT_TYPE = "org.apache.myfaces.orchestra.SeparateConversationContext";

    public void encodeBegin(FacesContext context) throws IOException
    {
        super.encodeBegin(context);

        // Stop appending the conversation context id to URLs. This means that a request using
        // those urls will trigger a new conversation context rather than inheriting the one
        // associated with this request.
        ConversationRequestParameterProvider.setInSeparationMode(true);
    }

    public void encodeChildren(FacesContext context) throws IOException
    {
        try
        {
            RendererUtils.renderChildren(context, this);
        }
        finally
        {
            // Restore the original separation mode. Actually, we just assume here that
            // the original value was false. There's no reason for anyone to ever nest
            // UISeparateConversationContext components, and nothing else fiddles with
            // this flag so false is a reasonable assumption.
            //
            // TODO: actually save the original value?
            ConversationRequestParameterProvider.setInSeparationMode(false);
        }
    }

    public boolean getRendersChildren()
    {
        return true;
    }

    public String getFamily()
    {
        return COMPONENT_FAMILY;
    }
}
