"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonacoEditorController = exports.MonacoEditorOperation = void 0;
var api_1 = require("@kie-tools-core/editor/dist/api");
var operating_system_1 = require("@kie-tools-core/operating-system");
var monaco_editor_1 = require("monaco-editor");
var MonacoEditorOperation;
(function (MonacoEditorOperation) {
    MonacoEditorOperation[MonacoEditorOperation["UNDO"] = 0] = "UNDO";
    MonacoEditorOperation[MonacoEditorOperation["REDO"] = 1] = "REDO";
    MonacoEditorOperation[MonacoEditorOperation["EDIT"] = 2] = "EDIT";
})(MonacoEditorOperation = exports.MonacoEditorOperation || (exports.MonacoEditorOperation = {}));
var MonacoEditorController = (function () {
    function MonacoEditorController(content, onContentChange, language, operatingSystem, isReadOnly, setValidationErrors) {
        var _this = this;
        this.onContentChange = onContentChange;
        this.language = language;
        this.operatingSystem = operatingSystem;
        this.isReadOnly = isReadOnly;
        this.setValidationErrors = setValidationErrors;
        this.getValidationMarkers = function () {
            return monaco_editor_1.editor.getModelMarkers({});
        };
        this.model = monaco_editor_1.editor.createModel(content, this.language);
        this.model.onDidChangeContent(function (event) {
            var _a;
            if (!event.isUndoing && !event.isRedoing) {
                (_a = _this.editor) === null || _a === void 0 ? void 0 : _a.pushUndoStop();
                onContentChange(_this.model.getValue(), MonacoEditorOperation.EDIT);
            }
        });
        monaco_editor_1.editor.onDidChangeMarkers(function () {
            _this.setValidationErrors(_this.getValidationMarkers());
        });
    }
    MonacoEditorController.prototype.redo = function () {
        var _a, _b;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.focus();
        (_b = this.editor) === null || _b === void 0 ? void 0 : _b.trigger("editor", "redo", null);
    };
    MonacoEditorController.prototype.undo = function () {
        var _a, _b;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.focus();
        (_b = this.editor) === null || _b === void 0 ? void 0 : _b.trigger("editor", "undo", null);
    };
    MonacoEditorController.prototype.setTheme = function (theme) {
        monaco_editor_1.editor.setTheme(this.getMonacoThemeByEditorTheme(theme));
    };
    MonacoEditorController.prototype.show = function (container, theme) {
        var _this = this;
        if (!container) {
            throw new Error("We need a container to show the editor!");
        }
        if (this.editor !== undefined) {
            this.setTheme(theme);
            return this.editor;
        }
        this.editor = monaco_editor_1.editor.create(container, {
            model: this.model,
            language: this.language,
            scrollBeyondLastLine: false,
            automaticLayout: true,
            fontSize: 12,
            theme: this.getMonacoThemeByEditorTheme(theme),
            readOnly: this.isReadOnly,
        });
        this.editor.addCommand(monaco_editor_1.KeyMod.CtrlCmd | monaco_editor_1.KeyCode.KeyZ, function () {
            _this.onContentChange(_this.model.getValue(), MonacoEditorOperation.UNDO);
        });
        this.editor.addCommand(monaco_editor_1.KeyMod.CtrlCmd | monaco_editor_1.KeyMod.Shift | monaco_editor_1.KeyCode.KeyZ, function () {
            _this.onContentChange(_this.model.getValue(), MonacoEditorOperation.REDO);
        });
        if (this.operatingSystem !== operating_system_1.OperatingSystem.MACOS) {
            this.editor.addCommand(monaco_editor_1.KeyMod.CtrlCmd | monaco_editor_1.KeyCode.KeyY, function () {
                _this.onContentChange(_this.model.getValue(), MonacoEditorOperation.REDO);
            });
        }
        return this.editor;
    };
    MonacoEditorController.prototype.getContent = function () {
        var _a, _b;
        return ((_b = (_a = this.editor) === null || _a === void 0 ? void 0 : _a.getModel()) === null || _b === void 0 ? void 0 : _b.getValue()) || "";
    };
    MonacoEditorController.prototype.forceRedraw = function () {
        var _a;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.render(true);
    };
    MonacoEditorController.prototype.getMonacoThemeByEditorTheme = function (theme) {
        switch (theme) {
            case api_1.EditorTheme.DARK:
                return "vs-dark";
            case api_1.EditorTheme.HIGH_CONTRAST:
                return "hc-black";
            default:
                return "vs";
        }
    };
    MonacoEditorController.prototype.dispose = function () {
        var _a;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.dispose();
    };
    return MonacoEditorController;
}());
exports.MonacoEditorController = MonacoEditorController;
//# sourceMappingURL=MonacoEditorController.js.map