/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.dht;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import java.math.BigInteger;
import java.util.Collection;
import java.util.Comparator;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.RangeStreamer;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.locator.EndpointsByRange;
import org.apache.cassandra.locator.EndpointsForRange;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.locator.Replica;
import org.apache.cassandra.locator.Replicas;
import org.psjava.algo.graph.flownetwork.FordFulkersonAlgorithm;
import org.psjava.algo.graph.flownetwork.MaximumFlowAlgorithm;
import org.psjava.algo.graph.flownetwork.MaximumFlowAlgorithmResult;
import org.psjava.algo.graph.pathfinder.DFSPathFinder;
import org.psjava.algo.graph.pathfinder.PathFinder;
import org.psjava.ds.graph.CapacityEdge;
import org.psjava.ds.graph.MutableCapacityGraph;
import org.psjava.ds.math.Function;
import org.psjava.ds.numbersystrem.AddableNumberSystem;
import org.psjava.ds.numbersystrem.IntegerNumberSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RangeFetchMapCalculator {
    private static final Logger logger = LoggerFactory.getLogger(RangeFetchMapCalculator.class);
    private static final long TRIVIAL_RANGE_LIMIT = 1000L;
    private final EndpointsByRange rangesWithSources;
    private final Predicate<Replica> sourceFilters;
    private final String keyspace;
    private final Vertex sourceVertex = OuterVertex.getSourceVertex();
    private final Vertex destinationVertex = OuterVertex.getDestinationVertex();
    private final Set<Range<Token>> trivialRanges;

    public RangeFetchMapCalculator(EndpointsByRange rangesWithSources, Collection<RangeStreamer.SourceFilter> sourceFilters, String keyspace) {
        this.rangesWithSources = rangesWithSources;
        this.sourceFilters = Predicates.and(sourceFilters);
        this.keyspace = keyspace;
        this.trivialRanges = rangesWithSources.keySet().stream().filter(RangeFetchMapCalculator::isTrivial).collect(Collectors.toSet());
    }

    static boolean isTrivial(Range<Token> range) {
        IPartitioner partitioner = DatabaseDescriptor.getPartitioner();
        if (partitioner.splitter().isPresent()) {
            BigInteger l = partitioner.splitter().get().valueForToken((Token)range.left);
            BigInteger r = partitioner.splitter().get().valueForToken((Token)range.right);
            if (r.compareTo(l) <= 0) {
                return false;
            }
            if (r.subtract(l).compareTo(BigInteger.valueOf(1000L)) < 0) {
                return true;
            }
        }
        return false;
    }

    public Multimap<InetAddressAndPort, Range<Token>> getRangeFetchMap() {
        HashMultimap fetchMap = HashMultimap.create();
        fetchMap.putAll(this.getRangeFetchMapForNonTrivialRanges());
        fetchMap.putAll(this.getRangeFetchMapForTrivialRanges((Multimap<InetAddressAndPort, Range<Token>>)fetchMap));
        return fetchMap;
    }

    @VisibleForTesting
    Multimap<InetAddressAndPort, Range<Token>> getRangeFetchMapForNonTrivialRanges() {
        MutableCapacityGraph<Vertex, Integer> graph = this.getGraph();
        this.addSourceAndDestination(graph, this.getDestinationLinkCapacity(graph));
        int flow = 0;
        MaximumFlowAlgorithmResult result = null;
        while (flow < this.getTotalRangeVertices(graph)) {
            if (flow > 0) {
                this.incrementCapacity(graph, 1);
            }
            MaximumFlowAlgorithm fordFulkerson = FordFulkersonAlgorithm.getInstance((PathFinder)DFSPathFinder.getInstance());
            result = fordFulkerson.calc(graph, (Object)this.sourceVertex, (Object)this.destinationVertex, (AddableNumberSystem)IntegerNumberSystem.getInstance());
            int newFlow = (Integer)result.calcTotalFlow();
            assert (newFlow > flow);
            flow = newFlow;
        }
        return this.getRangeFetchMapFromGraphResult(graph, result);
    }

    @VisibleForTesting
    Multimap<InetAddressAndPort, Range<Token>> getRangeFetchMapForTrivialRanges(Multimap<InetAddressAndPort, Range<Token>> optimisedMap) {
        HashMultimap fetchMap = HashMultimap.create();
        for (Range<Token> trivialRange : this.trivialRanges) {
            boolean added = false;
            boolean localDCCheck = true;
            while (!added) {
                EndpointsForRange replicas = (EndpointsForRange)this.rangesWithSources.get(trivialRange).sorted((Comparator)Comparator.comparingInt(o -> optimisedMap.get((Object)o.endpoint()).size()));
                Replicas.temporaryAssertFull(replicas);
                for (Replica replica : replicas) {
                    if (!this.passFilters(replica, localDCCheck)) continue;
                    added = true;
                    if (replica.isSelf()) continue;
                    fetchMap.put((Object)replica.endpoint(), trivialRange);
                    break;
                }
                if (!added && !localDCCheck) {
                    throw new IllegalStateException("Unable to find sufficient sources for streaming range " + trivialRange + " in keyspace " + this.keyspace);
                }
                if (!added) {
                    logger.info("Using other DC endpoints for streaming for range: {} and keyspace {}", trivialRange, (Object)this.keyspace);
                }
                localDCCheck = false;
            }
        }
        return fetchMap;
    }

    private int getTotalRangeVertices(MutableCapacityGraph<Vertex, Integer> graph) {
        int count = 0;
        for (Vertex vertex : graph.getVertices()) {
            if (!vertex.isRangeVertex()) continue;
            ++count;
        }
        return count;
    }

    private Multimap<InetAddressAndPort, Range<Token>> getRangeFetchMapFromGraphResult(MutableCapacityGraph<Vertex, Integer> graph, MaximumFlowAlgorithmResult<Integer, CapacityEdge<Vertex, Integer>> result) {
        HashMultimap rangeFetchMapMap = HashMultimap.create();
        if (result == null) {
            return rangeFetchMapMap;
        }
        Function flowFunction = result.calcFlowFunction();
        for (Vertex vertex : graph.getVertices()) {
            if (!vertex.isRangeVertex()) continue;
            boolean sourceFound = false;
            for (CapacityEdge e : graph.getEdges((Object)vertex)) {
                if ((Integer)flowFunction.get((Object)e) <= 0) continue;
                assert (!sourceFound);
                sourceFound = true;
                if (((Vertex)e.to()).isEndpointVertex()) {
                    rangeFetchMapMap.put((Object)((EndpointVertex)e.to()).getEndpoint(), ((RangeVertex)vertex).getRange());
                    continue;
                }
                if (!((Vertex)e.from()).isEndpointVertex()) continue;
                rangeFetchMapMap.put((Object)((EndpointVertex)e.from()).getEndpoint(), ((RangeVertex)vertex).getRange());
            }
            assert (sourceFound);
        }
        return rangeFetchMapMap;
    }

    private void incrementCapacity(MutableCapacityGraph<Vertex, Integer> graph, int incrementalCapacity) {
        for (Vertex vertex : graph.getVertices()) {
            if (!vertex.isEndpointVertex()) continue;
            graph.addEdge((Object)vertex, (Object)this.destinationVertex, (Object)incrementalCapacity);
        }
    }

    private void addSourceAndDestination(MutableCapacityGraph<Vertex, Integer> graph, int destinationCapacity) {
        graph.insertVertex((Object)this.sourceVertex);
        graph.insertVertex((Object)this.destinationVertex);
        for (Vertex vertex : graph.getVertices()) {
            if (vertex.isRangeVertex()) {
                graph.addEdge((Object)this.sourceVertex, (Object)vertex, (Object)1);
                continue;
            }
            if (!vertex.isEndpointVertex()) continue;
            graph.addEdge((Object)vertex, (Object)this.destinationVertex, (Object)destinationCapacity);
        }
    }

    private int getDestinationLinkCapacity(MutableCapacityGraph<Vertex, Integer> graph) {
        double endpointVertices = 0.0;
        double rangeVertices = 0.0;
        for (Vertex vertex : graph.getVertices()) {
            if (vertex.isEndpointVertex()) {
                endpointVertices += 1.0;
                continue;
            }
            if (!vertex.isRangeVertex()) continue;
            rangeVertices += 1.0;
        }
        return (int)Math.ceil(rangeVertices / endpointVertices);
    }

    private MutableCapacityGraph<Vertex, Integer> getGraph() {
        MutableCapacityGraph capacityGraph = MutableCapacityGraph.create();
        for (Range range : this.rangesWithSources.keySet()) {
            if (this.trivialRanges.contains(range)) {
                logger.debug("Not optimising trivial range {} for keyspace {}", (Object)range, (Object)this.keyspace);
                continue;
            }
            RangeVertex rangeVertex = new RangeVertex(range);
            boolean sourceFound = this.addEndpoints((MutableCapacityGraph<Vertex, Integer>)capacityGraph, rangeVertex, true);
            if (!sourceFound) {
                logger.info("Using other DC endpoints for streaming for range: {} and keyspace {}", (Object)range, (Object)this.keyspace);
                sourceFound = this.addEndpoints((MutableCapacityGraph<Vertex, Integer>)capacityGraph, rangeVertex, false);
            }
            if (sourceFound) continue;
            throw new IllegalStateException("Unable to find sufficient sources for streaming range " + range + " in keyspace " + this.keyspace);
        }
        return capacityGraph;
    }

    private boolean addEndpoints(MutableCapacityGraph<Vertex, Integer> capacityGraph, RangeVertex rangeVertex, boolean localDCCheck) {
        boolean sourceFound = false;
        Replicas.temporaryAssertFull(this.rangesWithSources.get(rangeVertex.getRange()));
        for (Replica replica : this.rangesWithSources.get(rangeVertex.getRange())) {
            if (!this.passFilters(replica, localDCCheck)) continue;
            sourceFound = true;
            if (replica.isSelf()) continue;
            EndpointVertex endpointVertex = new EndpointVertex(replica.endpoint());
            capacityGraph.insertVertex((Object)rangeVertex);
            capacityGraph.insertVertex((Object)endpointVertex);
            capacityGraph.addEdge((Object)rangeVertex, (Object)endpointVertex, (Object)Integer.MAX_VALUE);
        }
        return sourceFound;
    }

    private boolean isInLocalDC(Replica replica) {
        return DatabaseDescriptor.getLocalDataCenter().equals(DatabaseDescriptor.getEndpointSnitch().getDatacenter(replica));
    }

    private boolean passFilters(Replica replica, boolean localDCCheck) {
        return this.sourceFilters.apply((Object)replica) && (!localDCCheck || this.isInLocalDC(replica));
    }

    private static class OuterVertex
    extends Vertex {
        private final boolean source;

        private OuterVertex(boolean source) {
            this.source = source;
        }

        public static Vertex getSourceVertex() {
            return new OuterVertex(true);
        }

        public static Vertex getDestinationVertex() {
            return new OuterVertex(false);
        }

        @Override
        public Vertex.VERTEX_TYPE getVertexType() {
            return this.source ? Vertex.VERTEX_TYPE.SOURCE : Vertex.VERTEX_TYPE.DESTINATION;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            OuterVertex that = (OuterVertex)o;
            return this.source == that.source;
        }

        public int hashCode() {
            return this.source ? 1 : 0;
        }
    }

    private static class RangeVertex
    extends Vertex {
        private final Range<Token> range;

        public RangeVertex(Range<Token> range) {
            assert (range != null);
            this.range = range;
        }

        public Range<Token> getRange() {
            return this.range;
        }

        @Override
        public Vertex.VERTEX_TYPE getVertexType() {
            return Vertex.VERTEX_TYPE.RANGE;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            RangeVertex that = (RangeVertex)o;
            return this.range.equals(that.range);
        }

        public int hashCode() {
            return this.range.hashCode();
        }
    }

    private static class EndpointVertex
    extends Vertex {
        private final InetAddressAndPort endpoint;

        public EndpointVertex(InetAddressAndPort endpoint) {
            assert (endpoint != null);
            this.endpoint = endpoint;
        }

        public InetAddressAndPort getEndpoint() {
            return this.endpoint;
        }

        @Override
        public Vertex.VERTEX_TYPE getVertexType() {
            return Vertex.VERTEX_TYPE.ENDPOINT;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            EndpointVertex that = (EndpointVertex)o;
            return this.endpoint.equals(that.endpoint);
        }

        public int hashCode() {
            return this.endpoint.hashCode();
        }
    }

    private static abstract class Vertex {
        private Vertex() {
        }

        public abstract VERTEX_TYPE getVertexType();

        public boolean isEndpointVertex() {
            return this.getVertexType() == VERTEX_TYPE.ENDPOINT;
        }

        public boolean isRangeVertex() {
            return this.getVertexType() == VERTEX_TYPE.RANGE;
        }

        public static enum VERTEX_TYPE {
            ENDPOINT,
            RANGE,
            SOURCE,
            DESTINATION;

        }
    }
}

