/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.xmlbootstrap;

import org.junit.Assert;
import org.junit.Test;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;

import org.springframework.context.support.ClassPathXmlApplicationContext;

import test.common.beans.TestBean;


/**
 * JavaConfig must support user-defined names for {@link Configuration} beans bootstrapped via XML.
 *
 * @author  Chris Beams
 */
public class ConfigurationBeanNamePropagationTests {

    private static final String XML_NAME = ConfigurationBeanNamePropagationTests.class.getSimpleName().concat(".xml");

    @Test
    public void testPropagationOfNamedConfiguration() {
        ClassPathXmlApplicationContext ctx = new ClassPathXmlApplicationContext(XML_NAME, getClass());

        // this is the id of the @Configuration NamedConfig bean, explicitly set in XML
        String expectedName = "customConfigurationBean";

        // does the @Configuration bean have the name we expect?
        Assert.assertTrue("custom name did not survive processing", ctx.containsBean(expectedName));

        // should be exactly one NamedConfig bean.  An easy bug would be to accidentally create a
        // second NamedConfig bean.
        Assert.assertEquals("multiple instances of target config present",
                            1, ctx.getBeansOfType(NamedConfig.class).size());
    }

    @Test
    public void testPropagationOfUnnamedConfiguration() {
        ClassPathXmlApplicationContext ctx = new ClassPathXmlApplicationContext(XML_NAME, getClass());

        // by default, the name is the fully-qualified classname plus the index of the bean
        // (should always be 0)
        String expectedName = UnnamedConfig.class.getName().concat("#0");

        // there should be exactly one of this type of bean.
        Assert.assertEquals("multiple instances of target config present",
                            1, ctx.getBeansOfType(UnnamedConfig.class).size());

        // is it present?
        Assert.assertEquals("wrong name for @Configuration bean: ", expectedName,
                            ctx.getBeanNamesForType(UnnamedConfig.class)[0]);
    }

    public static class NamedConfig {
        @Bean
        public TestBean testBeanFromNamed() { return new TestBean(); }
    }

    public static class UnnamedConfig {
        @Bean
        public TestBean testBeanFromUnnamed() { return new TestBean(); }
    }

}
