/* Generated by re2c 0.13.5 */
/*
 * Copyright (c) 2014 Anders Wang Kristensen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "ujson.hpp"
#include "double-conversion.h"

#include <algorithm>
#include <sstream>

#ifdef __GNUC__
#ifdef __SSE2__
#define UJSON_USE_SSE2
#endif
#endif

#ifdef _MSC_VER
#if (defined(_M_AMD64) || defined(_M_X64))
#define UJSON_USE_SSE2
#elif _M_IX86_FP==2
#define UJSON_USE_SSE2
#endif
#endif

#ifdef UJSON_USE_SSE2
#include <emmintrin.h>
#endif

// vs2013 ctp 1 supports noexcept but rest don't
#if defined _MSC_VER && _MSC_FULL_VER != 180021114
#define noexcept
#endif

const ujson::value ujson::null = ujson::value();

// --------------------------------------------------------------------------
// utf-8
//
//    Code Points    |  1st   |  2nd  |  3rd  |  4th
//
// U+0000   - U+007F   00..7F
// U+0080   - U+07FF   C2..DF  80..BF
// U+0800   - U+0FFF   E0      A0..BF  80..BF
// U+1000   - U+CFFF   E1..EC  80..BF  80..BF
// U+D000   - U+D7FF   ED      80..9F  80..BF
// U+D800   - U+DFFF   illformed
// U+E000   - U+FFFF   EE..EF  80..BF  80..BF
// U+10000  - U+3FFFF  F0      90..BF  80..BF  80..BF
// U+40000  - U+FFFFF  F1..F3  80..BF  80..BF  80..BF
// U+100000 - U+10FFFF F4      80..8F  80..BF  80..BF

typedef std::pair<std::uint8_t, std::uint8_t> range_t;
struct utf8_ranges_t {
    std::uint8_t upper_bound;
    std::uint8_t num_ranges;
    range_t ranges[3];
};

static const utf8_ranges_t utf8_ranges[] = {
    { 0x7F, 0 },
    { 0xBF, 1 }, // 80-BF invalid continuation
    { 0xC1, 1 }, // 0xC0-0xC1 invalid as first byte
    { 0xDF, 1, { { 0x80, 0xBF + 1 } } },
    { 0xE0, 2, { { 0xA0, 0xBF + 1 }, { 0x80, 0xBF + 1 } } },
    { 0xEC, 2, { { 0x80, 0xBF + 1 }, { 0x80, 0xBF + 1 } } },
    { 0xED, 2, { { 0x80, 0x9F + 1 }, { 0x80, 0xBF + 1 } } },
    { 0xEF, 2, { { 0x80, 0xBF + 1 }, { 0x80, 0xBF + 1 } } },
    { 0xF0, 3, { { 0x90, 0xBF + 1 }, { 0x80, 0xBF + 1 }, { 0x80, 0xBF + 1 } } },
    { 0xF3, 3, { { 0x80, 0xBF + 1 }, { 0x80, 0xBF + 1 }, { 0x80, 0xBF + 1 } } },
    { 0xF4, 3, { { 0x80, 0x8F + 1 }, { 0x80, 0xBF + 1 }, { 0x80, 0xBF + 1 } } },
    { 0xFF, 1 } // 0xF5-0xFF invalid as first byte
};

static const std::size_t num_utf8_ranges =
    sizeof(utf8_ranges) / sizeof(utf8_ranges[0]);

bool ujson::value::is_valid_utf8(const char *ptr,
                                 const char *end) noexcept {
    while (ptr < end) {
#ifdef UJSON_USE_SSE2
        // test if the next 16 chars are ascii and skip forward as much as
        // possible
        while (end - ptr >= 16) {
            __m128i chunk =
                _mm_loadu_si128(reinterpret_cast<const __m128i *>(ptr));
            // signed comparison: multibyte utf-8 sequences have high bit set
            // are thus negative
            __m128i compare = _mm_cmplt_epi8(chunk, _mm_setzero_si128());
            int mask = _mm_movemask_epi8(compare);
            if (mask == 0) {
                ptr += 16;
                continue;
            }

#ifdef __GNUC__
            int index = __builtin_ffs(mask) - 1;
#else
            unsigned long index = 0;
            _BitScanForward(&index, mask);
#endif
            assert(index >= 0 && index <= 15);
            ptr += index;
            break;
        }
#endif
        auto c = static_cast<std::uint8_t>(*ptr++);
        auto it = utf8_ranges;
        for (; it != utf8_ranges + num_utf8_ranges; ++it) {
            if (c <= it->upper_bound)
                break;
        }
        assert(it < utf8_ranges + num_utf8_ranges);
        const auto limit = it->ranges + it->num_ranges;
        for (auto rng = it->ranges; rng < limit; ++rng) {
            auto d = static_cast<std::uint8_t>(*ptr++);
            if (d < rng->first || d >= rng->second)
                return false;
        }
    }
    return true;
}

// convert utf8 to utf32; returns utf32 + number of bytes consumed
static std::pair<std::uint32_t, std::uint32_t>
utf8_to_utf32(const char *ptr) {
    
    auto utf8 = reinterpret_cast<const std::uint8_t *>(ptr);
    
    std::uint8_t first = *utf8;
    std::uint32_t trailing_bytes;
    if (first <= 0x7F)
        trailing_bytes = 0;
    else if (first <= 0xDF)
        trailing_bytes = 1;
    else if (first <= 0xEF)
        trailing_bytes = 2;
    else
        trailing_bytes = 3;
    
    std::uint32_t utf32 = 0;
    switch (trailing_bytes) {
        case 3:
            utf32 += *utf8++;
            utf32 <<= 6;
        case 2:
            utf32 += *utf8++;
            utf32 <<= 6;
        case 1:
            utf32 += *utf8++;
            utf32 <<= 6;
        case 0:
            utf32 += *utf8++;
    }
    
    static const std::uint32_t magic[] = { 0x0, 0x3080, 0xE2080, 0x3C82080 };
    utf32 -= magic[trailing_bytes];
    assert(utf32 <= 0x10FFFF);
    return { utf32, trailing_bytes + 1 };
}

// convert utf32 code point to 1-4 bytes of utf8
static char *utf32_to_utf8(char *str, std::uint32_t cp) {
    
    assert(cp <= 0x10FFFF);
    
    std::size_t num_bytes;
    if (cp <= 0x7F)
        num_bytes = 1;
    else if (cp <= 0x7FF)
        num_bytes = 2;
    else if (cp <= 0xFFFF)
        num_bytes = 3;
    else
        num_bytes = 4;
    
    static const std::uint32_t offset[] = { 0x00, 0x00, 0xC0, 0xE0, 0xF0 };
    switch (num_bytes) {
        case 4:
            str[3] = static_cast<char>((cp | 0x80) & 0xBF);
            cp >>= 6;
        case 3:
            str[2] = static_cast<char>((cp | 0x80) & 0xBF);
            cp >>= 6;
        case 2:
            str[1] = static_cast<char>((cp | 0x80) & 0xBF);
            cp >>= 6;
        case 1:
            str[0] = static_cast<char>(cp | offset[num_bytes]);
    }
    
    return str + num_bytes;
}

// convert four byte hex string to int
static std::uint32_t hex_to_int(const std::uint8_t *hex) {
    
    static const std::uint8_t lookup[] = {
        0,  1,  2,  3,  4, 5, 6, 7, 8, 9, 0, 0,  0,  0,  0,  0,  0, 10, 11,
        12, 13, 14, 15, 0, 0, 0, 0, 0, 0, 0, 0,  0,  0,  0,  0,  0, 0,  0,
        0,  0,  0,  0,  0, 0, 0, 0, 0, 0, 0, 10, 11, 12, 13, 14, 15
    };
    
    assert(lookup['0' - '0'] == 0);
    assert(lookup['1' - '0'] == 1);
    assert(lookup['2' - '0'] == 2);
    assert(lookup['3' - '0'] == 3);
    assert(lookup['4' - '0'] == 4);
    assert(lookup['5' - '0'] == 5);
    assert(lookup['6' - '0'] == 6);
    assert(lookup['7' - '0'] == 7);
    assert(lookup['8' - '0'] == 8);
    assert(lookup['9' - '0'] == 9);
    assert(lookup['A' - '0'] == 10);
    assert(lookup['B' - '0'] == 11);
    assert(lookup['C' - '0'] == 12);
    assert(lookup['D' - '0'] == 13);
    assert(lookup['E' - '0'] == 14);
    assert(lookup['F' - '0'] == 15);
    assert(lookup['a' - '0'] == 10);
    assert(lookup['b' - '0'] == 11);
    assert(lookup['c' - '0'] == 12);
    assert(lookup['d' - '0'] == 13);
    assert(lookup['e' - '0'] == 14);
    assert(lookup['f' - '0'] == 15);
    
    std::uint32_t result = 0;
    for (int i = 0; i < 4; ++i) {
        result <<= 4;
        std::uint8_t c = hex[i];
        std::uint32_t h = lookup[c - '0'];
        result += h;
    }
    return result;
}

// convert int (<=0xFFFF) to six byte escaped hex string
static void int_to_hex(std::uint32_t cp, char *str) {
    assert(cp <= 0xFFFF);
    
    static const char *hex = "0123456789ABCDEF";
    
    str[0] = '\\';
    str[1] = 'u';
    str[2] = hex[(cp & 0xF000) >> 12];
    str[3] = hex[(cp & 0x0F00) >>  8];
    str[4] = hex[(cp & 0x00F0) >>  4];
    str[5] = hex[(cp & 0x000F) >>  0];
}

// ---------------------------------------------------------------------------
// to_string

static void to_string(std::string &result, ujson::string_view input,
                      const ujson::to_string_options &opts) {

    // upper bound on how much result can grow (happens if string consists of
    // certain control characters, since they are encoded as six characters,
    // \uXXXX)
    const auto max_size_increase = 6 * input.length() + 2;

    const auto old_size = result.size();
    result.resize(old_size + max_size_increase);

    char *out = &result[old_size];
    *out++ = '"';
    for (auto in = input.c_str(); in != input.c_str() + input.length(); ++in) {

        std::uint8_t c = *in;
        if (c >= 0x20) {

            if (c == '"') {
                *out++ = '\\';
                *out++ = '"';
            } else if (c == '\\') {
                *out++ = '\\';
                *out++ = '\\';
            } else if (c <= 127) {
                // ascii
                *out++ = c;
            } else if (opts.encoding == ujson::character_encoding::utf8) {
                // utf-8 multi-byte case
                *out++ = c;
            } else {

                // encode utf-8 multi-byte as utf-16
                auto pair = utf8_to_utf32(in);

                if (pair.first < 0x10000) {
                    int_to_hex(pair.first, out);
                    out += 6;
                } else {
                    std::uint32_t cp = pair.first - 0x10000;
                    std::uint32_t leading = (cp >> 10) + 0xD800;
                    assert(leading >= 0xD800 && leading < 0xDC00);
                    std::uint32_t trailing = (cp & 0x3FF) + 0xDC00;
                    assert(trailing >= 0xDC00 && trailing < 0xE000);

                    int_to_hex(leading, out);
                    out += 6;
                    int_to_hex(trailing, out);
                    out += 6;
                }

                in += pair.second - 1;
            }
        } else {
            if (c == '\b') {
                *out++ = '\\';
                *out++ = 'b';
            } else if (c == '\f') {
                *out++ = '\\';
                *out++ = 'f';
            } else if (c == '\n') {
                *out++ = '\\';
                *out++ = 'n';
            } else if (c == '\r') {
                *out++ = '\\';
                *out++ = 'r';
            } else if (c == '\t') {
                *out++ = '\\';
                *out++ = 't';
            } else {
                int_to_hex(c, out);
                out += 6;
            }
        }
    }
    *out++ = '"';

    // trim result to actual size
    const auto new_size = out - &result[0];
    assert(new_size - old_size <= max_size_increase);
    result.resize(new_size);
}

static void to_string(std::string &str, double v) {

    using namespace double_conversion;

    const int buffer_size = 128;
    char buffer[buffer_size];
    StringBuilder builder(buffer, buffer_size);

    auto flags = DoubleToStringConverter::NO_FLAGS;
    DoubleToStringConverter d2sc(flags, nullptr, nullptr, 'e', -10, 10, 0, 0);
#ifdef NDEBUG
    d2sc.ToShortest(v, &builder);
#else
    bool result = d2sc.ToShortest(v, &builder);
    assert(result);
#endif
    str += builder.Finalize();
}

static void to_string_impl(std::string &str, ujson::value const &v,
                           const ujson::to_string_options &opts,
                           std::size_t current_indent) {
    switch (v.type()) {
    case ujson::value_type::null:
        str += "null";
        break;
    case ujson::value_type::boolean:
        str += bool_cast(v) ? "true" : "false";
        break;
    case ujson::value_type::number:
        to_string(str, double_cast(v));
        break;
    case ujson::value_type::string:
        to_string(str, string_cast(v), opts);
        break;
    case ujson::value_type::array: {
        str += '[';
        if (opts.indent_amount > 0)
            str += '\n';
        auto const &array = array_cast(v);
        for (auto const &value : array) {
            str.append(current_indent + opts.indent_amount, ' ');
            to_string_impl(str, value, opts,
                           current_indent + opts.indent_amount);
            if (&value != &array.back())
                str += ',';
            if (opts.indent_amount > 0)
                str += '\n';
        }
        str.append(current_indent, ' ');
        str += ']';
        break;
    }
    case ujson::value_type::object: {
        str += '{';
        if (opts.indent_amount > 0)
            str += '\n';
        auto const &object = object_cast(v);
        bool first = true;
        for (auto const &kvp : object) {
            if (!first) {
                str += ',';
                if (opts.indent_amount > 0)
                    str += '\n';
            }
            str.append(current_indent + opts.indent_amount, ' ');
            to_string(str, { kvp.first.c_str(), kvp.first.length() }, opts);
            if (opts.indent_amount > 0)
                str += " : ";
            else
                str += ':';
            to_string_impl(str, kvp.second, opts,
                           current_indent + opts.indent_amount);
            first = false;
        }
        if (opts.indent_amount > 0 && !object.empty())
            str += '\n';
        str.append(current_indent, ' ');
        str += '}';
        break;
    }
    default:
        assert(false);
        break;
    }
}

std::string ujson::to_string(value const &v,
                             const ujson::to_string_options &opts) {
    std::string result;
    to_string_impl(result, v, opts, 0);
    return result;
}

std::ostream &ujson::operator<<(std::ostream &stream, value const &v) {
    stream << to_string(v);
    return stream;
}

//----------------------------------------------------------------------------
// exception

ujson::exception::exception(error_code error, int line)
    : m_error_code(error), m_line(line) {}

const char *ujson::exception::what() const noexcept{
    if (m_what.empty()) {
        std::stringstream ss;
        switch (m_error_code) {
        case error_code::bad_cast:
            ss << "Bad cast.";
            break;
        case error_code::bad_number:
            if (m_line == -1)
                ss << "Bad number.";
            else
                ss << "Bad number on line " << m_line << ".";
            break;
        case error_code::bad_string:
            ss << "Bad UTF-8.";
            break;
        case error_code::invalid_syntax:
            ss << "Invalid syntax on line " << m_line << ".";
            break;
        case error_code::integer_overflow:
            ss << "Number out of range for integer cast.";
            break;
        default:
            assert(false);
            break;
        }

        m_what = ss.str();
    }

    return m_what.c_str();
}

ujson::error_code ujson::exception::get_error_code() const {
    return m_error_code;
}

int ujson::exception::get_line() const { return m_line; }

//----------------------------------------------------------------------------
// parser

namespace {

enum token {
    ujson_colon,
    ujson_comma,
    ujson_null,
    ujson_true,
    ujson_false,
    ujson_number,
    ujson_string,
    ujson_array_begin,
    ujson_array_end,
    ujson_object_begin,
    ujson_object_end,
    ujson_eof
};

// helper class used for providing a sentinel token to the parser
// when it reads beyond the supplied buffer
class safe_ptr {
public:
    safe_ptr(const std::uint8_t *ptr, const std::uint8_t *limit);

    inline const std::uint8_t *ptr() const;
    inline const std::uint8_t *limit() const;

    inline std::uint8_t operator*() const;
    inline operator const std::uint8_t *() const;

    inline safe_ptr &operator++();
    inline safe_ptr &operator+=(std::size_t);

    inline safe_ptr &operator=(const std::uint8_t *);

private:
    const std::uint8_t *m_ptr;
    const std::uint8_t *m_limit;
};

class parser {
public:
    parser(const std::uint8_t *ptr, std::size_t len);

    token peek_token();
    token read_token();

    void expect(token token);

    double read_double() const;
    std::string read_string() const;

    int line() const;

private:
    token scan();

    const std::uint8_t *const m_start;
    const std::uint8_t *const m_limit;

    bool m_peeked;
    token m_current_token;

    safe_ptr m_cursor;
    const std::uint8_t *m_token;
};
}

//----------------------------------------------------------------------------

safe_ptr::safe_ptr(const std::uint8_t *ptr, const std::uint8_t *limit)
    : m_ptr(ptr), m_limit(limit) {
    assert(m_ptr <= m_limit);
}

const std::uint8_t *safe_ptr::ptr() const { return m_ptr; }
const std::uint8_t *safe_ptr::limit() const { return m_limit; }

std::uint8_t safe_ptr::operator*() const {
    return m_ptr < m_limit ? *m_ptr : 0;
}
safe_ptr::operator const std::uint8_t *() const { return m_ptr; }

safe_ptr &safe_ptr::operator++() {
    ++m_ptr;
    return *this;
}

safe_ptr &safe_ptr::operator+=(std::size_t offset) {
    m_ptr += offset;
    return *this;
}

safe_ptr &safe_ptr::operator=(const std::uint8_t *ptr) {
    m_ptr = ptr;
    return *this;
}

//----------------------------------------------------------------------------

parser::parser(const std::uint8_t *ptr, std::size_t len)
    : m_start(ptr), m_limit(ptr + len), m_cursor(ptr, ptr + len) {
    m_peeked = false;
}

int parser::line() const {
    return static_cast<int>(std::count(m_start, m_cursor.ptr(), '\n') + 1);
}

token parser::peek_token() {
    if (!m_peeked) {
        m_current_token = scan();
        m_peeked = true;
    }

    return m_current_token;
}

token parser::read_token() {
    if (!m_peeked)
        m_current_token = scan();

    m_peeked = false;
    return m_current_token;
}

void parser::expect(token token) {
    if (token != read_token())
        throw ujson::exception(ujson::error_code::invalid_syntax, line());
}

double parser::read_double() const {

    const auto len = static_cast<int>(m_cursor.ptr() - m_token);

    using namespace double_conversion;
    auto flags = StringToDoubleConverter::NO_FLAGS;
    StringToDoubleConverter s2dc(flags, 0.0, 0.0, nullptr, nullptr);
    int processed_chars;
    auto token = reinterpret_cast<const char *>(m_token);
    double result = s2dc.StringToDouble(token, len, &processed_chars);

    // handle invalid number
    if (processed_chars != len)
        throw ujson::exception(ujson::error_code::bad_number, line());

    // handle overflow
    if (!std::isfinite(result))
        throw ujson::exception(ujson::error_code::bad_number, line());
    return result;
}

std::string parser::read_string() const {

    // m_token points to first double qoute and m_cursor points to last
    auto in = m_token + 1;
    const auto limit = m_cursor.ptr() - 1;
    if (in == limit)
        return "";

    // limit-in is an upper bound on the size of the resulting string
    std::string result(limit - in, '\0');
    char *out = &result.front();

    while (in < limit) {

#ifdef UJSON_USE_SSE2
        while (limit - in >= 16) {
            __m128i backslash = _mm_set1_epi8(0x5C);
            __m128i chunk =
                _mm_loadu_si128(reinterpret_cast<const __m128i *>(in));
            __m128i compare = _mm_cmpeq_epi8(chunk, backslash);
            int mask = _mm_movemask_epi8(compare);
            if (!mask) {
                in += 16;
                _mm_storeu_si128(reinterpret_cast<__m128i *>(out), chunk);
                out += 16;
                continue;
            }

#ifdef __GNUC__
            int index = __builtin_ffs(mask) - 1;
#else
            unsigned long index = 0;
            _BitScanForward(&index, mask);
#endif
            assert(index >= 0 && index <= 15);
            for (decltype(index) i = 0; i < index; ++i)
                *out++ = *in++;
            break;
        }

        if (in == limit)
            break;

#endif
        char c = *in++;
        if (c != '\\') {
            *out++ = c;
        } else {
            c = *in++;
            if (c == '\\')
                *out++ = '\\';
            else if (c == '"')
                *out++ = '"';
            else if (c == '/')
                *out++ = '/';
            else if (c == 'b')
                *out++ = '\b';
            else if (c == 'f')
                *out++ = '\f';
            else if (c == 'n')
                *out++ = '\n';
            else if (c == 'r')
                *out++ = '\r';
            else if (c == 't')
                *out++ = '\t';
            else if (c == 'u') {
                std::uint32_t cp = hex_to_int(in);
                in += 4;

                // surrogate pair?
                if (cp >= 0xD800 && cp <= 0xDBFF) {
                    in += 2;
                    std::uint32_t trailing = hex_to_int(in);
                    in += 4;
                    assert(trailing >= 0xDC00 && trailing <= 0xDFFF);
                    cp =
                        ((cp - 0xD800) << 10) + (trailing - 0xDC00) + 0x10000;
                }

                // translate to 1-4 utf-8 chars
                out = utf32_to_utf8(out, cp);
            } else {
                // can't happen unless regexes are wrong
                assert(false);
            }
        }
    }

    result.resize(out - result.data());
    return result;
}

//----------------------------------------------------------------------------

token parser::scan() {
    safe_ptr marker(m_cursor, m_limit);

std:
#ifdef UJSON_USE_SSE2
    while (m_limit - m_cursor.ptr() >= 16) {

        __m128i chunk = _mm_loadu_si128(
            reinterpret_cast<const __m128i *>(m_cursor.ptr()));

        __m128i tabs = _mm_set1_epi8(0x09);
        __m128i newlines = _mm_set1_epi8(0x0A);
        __m128i carriage_return = _mm_set1_epi8(0x0D);
        __m128i spaces = _mm_set1_epi8(0x20);

        __m128i is_tabs = _mm_cmpeq_epi8(chunk, tabs);
        __m128i is_newlines = _mm_cmpeq_epi8(chunk, newlines);
        __m128i is_carriage_return = _mm_cmpeq_epi8(chunk, carriage_return);
        __m128i is_space = _mm_cmpeq_epi8(chunk, spaces);

        __m128i is_white_space =
            _mm_or_si128(_mm_or_si128(is_tabs, is_newlines),
                         _mm_or_si128(is_carriage_return, is_space));

        int mask = _mm_movemask_epi8(is_white_space);
        if (mask == 0xFFFF) {
            m_cursor += 16;
            continue;
        }

#ifdef __GNUC__
        m_cursor += __builtin_ffs(~mask) - 1;
#else
        unsigned long index = 0;
        if (_BitScanForward(&index, ~mask))
            m_cursor += index;
#endif
        break;
    }
#endif

    m_token = m_cursor.ptr();

    
{
    std::uint8_t yych;
    unsigned int yyaccept = 0;

    yych = *m_cursor;
    switch (yych) {
    case 0x00:    goto ujson25;
    case '\t':
    case '\n':
    case '\r':
    case ' ':    goto ujson23;
    case '"':    goto ujson22;
    case ',':    goto ujson4;
    case '-':    goto ujson18;
    case '0':    goto ujson19;
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':    goto ujson21;
    case ':':    goto ujson2;
    case '[':    goto ujson10;
    case ']':    goto ujson12;
    case 'f':    goto ujson9;
    case 'n':    goto ujson6;
    case 't':    goto ujson8;
    case '{':    goto ujson14;
    case '}':    goto ujson16;
    default:    goto ujson27;
    }
ujson2:
    ++m_cursor;
    { return ujson_colon; }
ujson4:
    ++m_cursor;
    { return ujson_comma; }
ujson6:
    yyaccept = 0;
    yych = *(marker = ++m_cursor);
    switch (yych) {
    case 'u':    goto ujson91;
    default:    goto ujson7;
    }
ujson7:
    {
        throw ujson::exception(ujson::error_code::invalid_syntax, line());
     }
ujson8:
    yyaccept = 0;
    yych = *(marker = ++m_cursor);
    switch (yych) {
    case 'r':    goto ujson87;
    default:    goto ujson7;
    }
ujson9:
    yyaccept = 0;
    yych = *(marker = ++m_cursor);
    switch (yych) {
    case 'a':    goto ujson82;
    default:    goto ujson7;
    }
ujson10:
    ++m_cursor;
    { return ujson_array_begin; }
ujson12:
    ++m_cursor;
    { return ujson_array_end; }
ujson14:
    ++m_cursor;
    { return ujson_object_begin; }
ujson16:
    ++m_cursor;
    { return ujson_object_end; }
ujson18:
    yych = *++m_cursor;
    switch (yych) {
    case '0':    goto ujson81;
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':    goto ujson72;
    default:    goto ujson7;
    }
ujson19:
    yyaccept = 1;
    yych = *(marker = ++m_cursor);
    switch (yych) {
    case '.':    goto ujson74;
    case 'E':
    case 'e':    goto ujson75;
    default:    goto ujson20;
    }
ujson20:
    { return ujson_number; }
ujson21:
    yyaccept = 1;
    yych = *(marker = ++m_cursor);
    goto ujson73;
ujson22:
    yyaccept = 0;
    yych = *(marker = ++m_cursor);
    switch (yych) {
    case ' ':
    case '!':
    case '"':
    case '#':
    case '$':
    case '%':
    case '&':
    case '\'':
    case '(':
    case ')':
    case '*':
    case '+':
    case ',':
    case '-':
    case '.':
    case '/':
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case ':':
    case ';':
    case '<':
    case '=':
    case '>':
    case '?':
    case '@':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'G':
    case 'H':
    case 'I':
    case 'J':
    case 'K':
    case 'L':
    case 'M':
    case 'N':
    case 'O':
    case 'P':
    case 'Q':
    case 'R':
    case 'S':
    case 'T':
    case 'U':
    case 'V':
    case 'W':
    case 'X':
    case 'Y':
    case 'Z':
    case '[':
    case '\\':
    case ']':
    case '^':
    case '_':
    case '`':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':
    case 'g':
    case 'h':
    case 'i':
    case 'j':
    case 'k':
    case 'l':
    case 'm':
    case 'n':
    case 'o':
    case 'p':
    case 'q':
    case 'r':
    case 's':
    case 't':
    case 'u':
    case 'v':
    case 'w':
    case 'x':
    case 'y':
    case 'z':
    case '{':
    case '|':
    case '}':
    case '~':
    case 0x7F:
    case 0xC2:
    case 0xC3:
    case 0xC4:
    case 0xC5:
    case 0xC6:
    case 0xC7:
    case 0xC8:
    case 0xC9:
    case 0xCA:
    case 0xCB:
    case 0xCC:
    case 0xCD:
    case 0xCE:
    case 0xCF:
    case 0xD0:
    case 0xD1:
    case 0xD2:
    case 0xD3:
    case 0xD4:
    case 0xD5:
    case 0xD6:
    case 0xD7:
    case 0xD8:
    case 0xD9:
    case 0xDA:
    case 0xDB:
    case 0xDC:
    case 0xDD:
    case 0xDE:
    case 0xDF:
    case 0xE0:
    case 0xE1:
    case 0xE2:
    case 0xE3:
    case 0xE4:
    case 0xE5:
    case 0xE6:
    case 0xE7:
    case 0xE8:
    case 0xE9:
    case 0xEA:
    case 0xEB:
    case 0xEC:
    case 0xED:
    case 0xEE:
    case 0xEF:
    case 0xF0:
    case 0xF1:
    case 0xF2:
    case 0xF3:
    case 0xF4:    goto ujson31;
    default:    goto ujson7;
    }
ujson23:
    ++m_cursor;
    yych = *m_cursor;
    goto ujson29;
ujson24:
    { goto std; }
ujson25:
    ++m_cursor;
    { return ujson_eof; }
ujson27:
    yych = *++m_cursor;
    goto ujson7;
ujson28:
    ++m_cursor;
    yych = *m_cursor;
ujson29:
    switch (yych) {
    case '\t':
    case '\n':
    case '\r':
    case ' ':    goto ujson28;
    default:    goto ujson24;
    }
ujson30:
    ++m_cursor;
    yych = *m_cursor;
ujson31:
    switch (yych) {
    case ' ':
    case '!':
    case '#':
    case '$':
    case '%':
    case '&':
    case '\'':
    case '(':
    case ')':
    case '*':
    case '+':
    case ',':
    case '-':
    case '.':
    case '/':
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case ':':
    case ';':
    case '<':
    case '=':
    case '>':
    case '?':
    case '@':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'G':
    case 'H':
    case 'I':
    case 'J':
    case 'K':
    case 'L':
    case 'M':
    case 'N':
    case 'O':
    case 'P':
    case 'Q':
    case 'R':
    case 'S':
    case 'T':
    case 'U':
    case 'V':
    case 'W':
    case 'X':
    case 'Y':
    case 'Z':
    case '[':
    case ']':
    case '^':
    case '_':
    case '`':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':
    case 'g':
    case 'h':
    case 'i':
    case 'j':
    case 'k':
    case 'l':
    case 'm':
    case 'n':
    case 'o':
    case 'p':
    case 'q':
    case 'r':
    case 's':
    case 't':
    case 'u':
    case 'v':
    case 'w':
    case 'x':
    case 'y':
    case 'z':
    case '{':
    case '|':
    case '}':
    case '~':
    case 0x7F:    goto ujson30;
    case '"':    goto ujson42;
    case '\\':    goto ujson33;
    case 0xC2:
    case 0xC3:
    case 0xC4:
    case 0xC5:
    case 0xC6:
    case 0xC7:
    case 0xC8:
    case 0xC9:
    case 0xCA:
    case 0xCB:
    case 0xCC:
    case 0xCD:
    case 0xCE:
    case 0xCF:
    case 0xD0:
    case 0xD1:
    case 0xD2:
    case 0xD3:
    case 0xD4:
    case 0xD5:
    case 0xD6:
    case 0xD7:
    case 0xD8:
    case 0xD9:
    case 0xDA:
    case 0xDB:
    case 0xDC:
    case 0xDD:
    case 0xDE:
    case 0xDF:    goto ujson34;
    case 0xE0:    goto ujson35;
    case 0xE1:
    case 0xE2:
    case 0xE3:
    case 0xE4:
    case 0xE5:
    case 0xE6:
    case 0xE7:
    case 0xE8:
    case 0xE9:
    case 0xEA:
    case 0xEB:
    case 0xEC:    goto ujson36;
    case 0xED:    goto ujson37;
    case 0xEE:
    case 0xEF:    goto ujson38;
    case 0xF0:    goto ujson39;
    case 0xF1:
    case 0xF2:
    case 0xF3:    goto ujson40;
    case 0xF4:    goto ujson41;
    default:    goto ujson32;
    }
ujson32:
    m_cursor = marker;
    switch (yyaccept) {
    case 0: 	goto ujson7;
    case 1: 	goto ujson20;
    }
ujson33:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '"':
    case '/':
    case '\\':
    case 'b':
    case 'f':
    case 'n':
    case 'r':
    case 't':    goto ujson30;
    case 'u':    goto ujson54;
    default:    goto ujson32;
    }
ujson34:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson30;
    default:    goto ujson32;
    }
ujson35:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson53;
    default:    goto ujson32;
    }
ujson36:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson52;
    default:    goto ujson32;
    }
ujson37:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:    goto ujson51;
    default:    goto ujson32;
    }
ujson38:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson50;
    default:    goto ujson32;
    }
ujson39:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson48;
    default:    goto ujson32;
    }
ujson40:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson46;
    default:    goto ujson32;
    }
ujson41:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:    goto ujson44;
    default:    goto ujson32;
    }
ujson42:
    ++m_cursor;
    { return ujson_string; }
ujson44:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson45;
    default:    goto ujson32;
    }
ujson45:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson30;
    default:    goto ujson32;
    }
ujson46:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson47;
    default:    goto ujson32;
    }
ujson47:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson30;
    default:    goto ujson32;
    }
ujson48:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson49;
    default:    goto ujson32;
    }
ujson49:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson30;
    default:    goto ujson32;
    }
ujson50:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson30;
    default:    goto ujson32;
    }
ujson51:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson30;
    default:    goto ujson32;
    }
ujson52:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson30;
    default:    goto ujson32;
    }
ujson53:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 0x80:
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
    case 0x89:
    case 0x8A:
    case 0x8B:
    case 0x8C:
    case 0x8D:
    case 0x8E:
    case 0x8F:
    case 0x90:
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
    case 0x99:
    case 0x9A:
    case 0x9B:
    case 0x9C:
    case 0x9D:
    case 0x9E:
    case 0x9F:
    case 0xA0:
    case 0xA1:
    case 0xA2:
    case 0xA3:
    case 0xA4:
    case 0xA5:
    case 0xA6:
    case 0xA7:
    case 0xA8:
    case 0xA9:
    case 0xAA:
    case 0xAB:
    case 0xAC:
    case 0xAD:
    case 0xAE:
    case 0xAF:
    case 0xB0:
    case 0xB1:
    case 0xB2:
    case 0xB3:
    case 0xB4:
    case 0xB5:
    case 0xB6:
    case 0xB7:
    case 0xB8:
    case 0xB9:
    case 0xBA:
    case 0xBB:
    case 0xBC:
    case 0xBD:
    case 0xBE:
    case 0xBF:    goto ujson30;
    default:    goto ujson32;
    }
ujson54:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'a':
    case 'b':
    case 'c':    goto ujson55;
    case 'D':
    case 'd':    goto ujson56;
    case 'E':
    case 'F':
    case 'e':
    case 'f':    goto ujson57;
    default:    goto ujson32;
    }
ujson55:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson70;
    default:    goto ujson32;
    }
ujson56:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':    goto ujson61;
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'a':
    case 'b':    goto ujson60;
    default:    goto ujson32;
    }
ujson57:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson58;
    default:    goto ujson32;
    }
ujson58:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson59;
    default:    goto ujson32;
    }
ujson59:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson30;
    default:    goto ujson32;
    }
ujson60:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson63;
    default:    goto ujson32;
    }
ujson61:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson62;
    default:    goto ujson32;
    }
ujson62:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson30;
    default:    goto ujson32;
    }
ujson63:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson64;
    default:    goto ujson32;
    }
ujson64:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '\\':    goto ujson65;
    default:    goto ujson32;
    }
ujson65:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 'u':    goto ujson66;
    default:    goto ujson32;
    }
ujson66:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 'D':
    case 'd':    goto ujson67;
    default:    goto ujson32;
    }
ujson67:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson68;
    default:    goto ujson32;
    }
ujson68:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson69;
    default:    goto ujson32;
    }
ujson69:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson30;
    default:    goto ujson32;
    }
ujson70:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson71;
    default:    goto ujson32;
    }
ujson71:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':    goto ujson30;
    default:    goto ujson32;
    }
ujson72:
    yyaccept = 1;
    marker = ++m_cursor;
    yych = *m_cursor;
ujson73:
    switch (yych) {
    case '.':    goto ujson74;
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':    goto ujson72;
    case 'E':
    case 'e':    goto ujson75;
    default:    goto ujson20;
    }
ujson74:
    yych = *++m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':    goto ujson79;
    default:    goto ujson32;
    }
ujson75:
    yych = *++m_cursor;
    switch (yych) {
    case '+':
    case '-':    goto ujson76;
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':    goto ujson77;
    default:    goto ujson32;
    }
ujson76:
    yych = *++m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':    goto ujson77;
    default:    goto ujson32;
    }
ujson77:
    ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':    goto ujson77;
    default:    goto ujson20;
    }
ujson79:
    yyaccept = 1;
    marker = ++m_cursor;
    yych = *m_cursor;
    switch (yych) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':    goto ujson79;
    case 'E':
    case 'e':    goto ujson75;
    default:    goto ujson20;
    }
ujson81:
    yyaccept = 1;
    yych = *(marker = ++m_cursor);
    switch (yych) {
    case '.':    goto ujson74;
    case 'E':
    case 'e':    goto ujson75;
    default:    goto ujson20;
    }
ujson82:
    yych = *++m_cursor;
    switch (yych) {
    case 'l':    goto ujson83;
    default:    goto ujson32;
    }
ujson83:
    yych = *++m_cursor;
    switch (yych) {
    case 's':    goto ujson84;
    default:    goto ujson32;
    }
ujson84:
    yych = *++m_cursor;
    switch (yych) {
    case 'e':    goto ujson85;
    default:    goto ujson32;
    }
ujson85:
    ++m_cursor;
    { return ujson_false; }
ujson87:
    yych = *++m_cursor;
    switch (yych) {
    case 'u':    goto ujson88;
    default:    goto ujson32;
    }
ujson88:
    yych = *++m_cursor;
    switch (yych) {
    case 'e':    goto ujson89;
    default:    goto ujson32;
    }
ujson89:
    ++m_cursor;
    { return ujson_true; }
ujson91:
    yych = *++m_cursor;
    switch (yych) {
    case 'l':    goto ujson92;
    default:    goto ujson32;
    }
ujson92:
    yych = *++m_cursor;
    switch (yych) {
    case 'l':    goto ujson93;
    default:    goto ujson32;
    }
ujson93:
    ++m_cursor;
    { return ujson_null; }
}

}

//----------------------------------------------------------------------------

static ujson::value parse_value(parser &parser) {
    switch (parser.peek_token()) {
    case ujson_null:
        parser.read_token();
        return ujson::null;
    case ujson_true:
        parser.read_token();
        return true;
    case ujson_false:
        parser.read_token();
        return false;
    case ujson_number:
        parser.read_token();
        return parser.read_double();
    case ujson_string: {
        parser.read_token();
        auto string = parser.read_string();
        return ujson::value(std::move(string), ujson::validate_utf8::no);
    }
    case ujson_array_begin: {
        parser.read_token();
        ujson::array array;
        bool first = true;
        while (parser.peek_token() != ujson_array_end) {
            if (!first)
                parser.expect(ujson_comma);
            auto value = parse_value(parser);
            array.push_back(std::move(value));
            first = false;
        }
        parser.read_token();
        return ujson::value(std::move(array));
    }
    case ujson_object_begin: {
        parser.read_token();
        ujson::object object;
        bool first = true;
        while (parser.peek_token() != ujson_object_end) {
            if (!first)
                parser.expect(ujson_comma);
            parser.expect(ujson_string);
            auto key = parser.read_string();
            parser.expect(ujson_colon);
            auto value = parse_value(parser);
            object.emplace_back(std::move(key), std::move(value));
            first = false;
        }
        parser.read_token();
        return ujson::value(std::move(object), ujson::validate_utf8::no);
    }
    default:
        throw ujson::exception(ujson::error_code::invalid_syntax,
                               parser.line());
    }
}

ujson::value ujson::parse(const std::string &str) {
    return parse(str.c_str(), str.size());
}

ujson::value ujson::parse(const char *buffer, std::size_t len) {

    auto buf = reinterpret_cast<const std::uint8_t *>(buffer);
    parser parser(buf, len ? len : std::strlen(buffer));
    auto result = parse_value(parser);

    // fail if trailing junk is found
    if (parser.read_token() != ujson_eof)
        throw ujson::exception(ujson::error_code::invalid_syntax, parser.line());
    return result;
}
