package org.springframework.uaa.client.internal;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.UUID;
import java.util.prefs.Preferences;

import org.springframework.uaa.client.ProxyService;
import org.springframework.uaa.client.TransmissionService;
import org.springframework.uaa.client.util.Assert;
import org.springframework.uaa.client.util.PreferencesUtils;
import org.springframework.uaa.client.util.StreamUtils;

/**
 * Basic implementation of {@link TransmissionService} that uses the {@link URL} object.
 * 
 * @author Ben Alex
 * @author Christian Dupuis
 * @since 1.0.1
 */
public class JdkUrlTransmissionServiceImpl implements TransmissionService {
	
	static final String BUCKET_NAME_KEY = "bucket_name";
	static final String PATH_NAME_KEY = "path_name";
	static final String POLICY_KEY = "policy";
	static final String SIGNATURE_KEY = "signature";
	static final String ACCESS_KEY = "aws_access_key";
	private ProxyService proxyService;
	
	private static final Preferences P = PreferencesUtils.getPreferencesFor(JdkUrlTransmissionServiceImpl.class);

	/**
	 * Constructs {@link JdkUrlTransmissionServiceImpl} using the passed {@link ProxyService} for proxy configuration.
	 * 
	 * @param proxyService to use for proxy-related configuration (required)
	 */
	public JdkUrlTransmissionServiceImpl(ProxyService proxyService) {
		Assert.notNull(proxyService, "Proxy service required");
		this.proxyService = proxyService;
	}
	
	/**
	 * Downloads the given {@link URL} resource.
	 * 
	 * @param url the {@link URL} to download
	 * @return stream to access the resource
	 */
	public InputStream download(URL url) throws IOException {
		if (url == null) throw new IllegalArgumentException("URL to download required");
		return proxyService.prepareHttpUrlConnection(url).getInputStream();
	}
	
	/**
	 * Uploads the passed file to http://uaa.springsource.org/uploads, giving a
	 * unique UUID for the upload. Only files of a byte size between 1 and
	 * 1,048,576 (ie 1 MB) are acceptable. The resulting file will be "private",
	 * meaning it can only be opened by the bucket owner.
	 * 
	 * @param inputStream to upload (cannot be null)
	 * @throws IOException if there is an I/O error
	 * @return <code>true</code> indicating successful upload
	 */
	public boolean upload(InputStream inputStream) throws IOException {
		if (inputStream == null) throw new IllegalArgumentException("Input stream to transmit required");
		String boundary = "------------" + UUID.randomUUID().toString();

		URL uaaUrl = new URL(P.get(BUCKET_NAME_KEY, "http://uaa.springsource.org/"));
		 
		HttpURLConnection connection = proxyService.prepareHttpUrlConnection(uaaUrl);
		connection.setDoOutput(true);
		connection.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + boundary);
		OutputStream output = connection.getOutputStream();

		PrintWriter writer = new PrintWriter(output, true);

		/*
		Use http://s3.amazonaws.com/doc/s3-example-code/post/post_sample.html to change the plain text policy:
		{ "expiration":
		   "2019-12-31T12:00:00.000Z", 
		   "conditions": [ {"bucket": "uaa.springsource.org" }, {"acl": "private" }, ["starts-with", "$key", "uploads/"], ["content-length-range", 1, 1048576] ]
		}
		*/
		append("key", P.get(PATH_NAME_KEY, "uploads/") + UUID.randomUUID().toString(), boundary, writer);
		append("acl", "private", boundary, writer);
		append("AWSAccessKeyId", P.get(ACCESS_KEY, "0SCA5K7NAW330XGHMT02"), boundary, writer);
		append("policy", P.get(POLICY_KEY, "ewogICJleHBpcmF0aW9uIjogIjIwMTktMTItMzFUMTI6MDA6MDAuMDAwWiIsCiAgImNvbmRpdGlvbnMiOiBbCiAgICB7ImJ1Y2tldCI6ICJ1YWEuc3ByaW5nc291cmNlLm9yZyIgfSwKICAgIHsiYWNsIjogInByaXZhdGUiIH0sCiAgICBbInN0YXJ0cy13aXRoIiwgIiRrZXkiLCAidXBsb2Fkcy8iXSwKICAgIFsiY29udGVudC1sZW5ndGgtcmFuZ2UiLCAxLCAxMDQ4NTc2XQogIF0KfQo="), boundary, writer);
		append("signature", P.get(SIGNATURE_KEY, "ZuVAJZxCiBzErDB2Jm4n/WvEpSE="), boundary, writer);

		println(writer, "--" + boundary);
		println(writer, "Content-Disposition: form-data; name=\"file\"; filename=\"not_important\"");
		println(writer, "Content-Type: application/octet-stream");
		println(writer, "");
		StreamUtils.copy(inputStream, output);
		println(writer, "");
		println(writer, "--" + boundary + "--");
		output.flush();
		close(inputStream, writer, output);
		return connection.getResponseCode() == 204;
	}

	private void append(String key, String value, String boundary, PrintWriter writer) {
		println(writer, "--" + boundary);
		println(writer, "Content-Disposition: form-data; name=\"" + key + "\"");
		println(writer, "");
		println(writer, value);
	}

	private void close(Closeable... objs) {
		for (Closeable closeable : objs) {
			try {
				closeable.close();
			} catch (Exception ignore) {
			}
		}
	}

	/**
	 * Prints a line that ends in <code>\r\n</code>, being the requirement for multipart uploads.
	 * If an empty string is passed, it simply outputs a single <code>\r\n</code>.
	 * 
	 * @param writer to use to emit the line
	 * @param line the line (do not include any CR/LF characters)
	 */
	private void println(PrintWriter writer, String line) {
		if (!("".equals(line))) {
			writer.print(line);
		}
		writer.print("\r\n");
		writer.flush();
	}

}
