/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.project.extensions;

import docking.widgets.OkDialog;
import docking.widgets.OptionDialog;
import generic.jar.ResourceFile;
import ghidra.framework.Application;
import ghidra.framework.project.extensions.ExtensionDetails;
import ghidra.util.Msg;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskLauncher;
import ghidra.util.task.TaskMonitor;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.compress.utils.IOUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import utilities.util.FileUtilities;
import utility.application.ApplicationLayout;

public class ExtensionUtils {
    private static final int ZIPFILE = 1347093252;
    public static String PROPERTIES_FILE_NAME = "extension.properties";
    public static String PROPERTIES_FILE_NAME_UNINSTALLED = "extension.properties.uninstalled";
    private static final Logger log = LogManager.getLogger(ExtensionUtils.class);

    public static void initializeExtensions() {
        Extensions extensions = ExtensionUtils.getAllInstalledExtensions();
        extensions.cleanupExtensionsMarkedForRemoval();
        extensions.reportDuplicateExtensions();
    }

    public static Set<ExtensionDetails> getActiveInstalledExtensions() {
        Extensions extensions = ExtensionUtils.getAllInstalledExtensions();
        return extensions.getActiveExtensions();
    }

    public static Set<ExtensionDetails> getInstalledExtensions() {
        Extensions extensions = ExtensionUtils.getAllInstalledExtensions();
        return extensions.get();
    }

    private static Extensions getAllInstalledExtensions() {
        log.trace("Finding all installed extensions...");
        Extensions extensions = new Extensions();
        ApplicationLayout layout = Application.getApplicationLayout();
        for (ResourceFile installDir : layout.getExtensionInstallationDirs()) {
            if (!installDir.isDirectory()) continue;
            log.trace("Checking extension installation dir '" + installDir);
            File dir = installDir.getFile(false);
            List<File> propFiles = ExtensionUtils.findExtensionPropertyFiles(dir);
            for (File propFile : propFiles) {
                ExtensionDetails extension = ExtensionUtils.createExtensionFromProperties(propFile);
                if (extension == null) continue;
                File extInstallDir = propFile.getParentFile();
                extension.setInstallDir(extInstallDir);
                log.trace("Loading extension '" + extension.getName() + "' from: " + extInstallDir);
                extensions.add(extension);
            }
        }
        log.trace(() -> "All installed extensions: " + extensions.getAsString());
        return extensions;
    }

    public static Set<ExtensionDetails> getArchiveExtensions() {
        log.trace("Finding archived extensions");
        ApplicationLayout layout = Application.getApplicationLayout();
        ResourceFile archiveDir = layout.getExtensionArchiveDir();
        if (archiveDir == null) {
            log.trace("No extension archive dir found");
            return Collections.emptySet();
        }
        ResourceFile[] archiveFiles = archiveDir.listFiles();
        if (archiveFiles == null) {
            log.trace("No files in extension archive dir: " + archiveDir);
            return Collections.emptySet();
        }
        HashSet<ExtensionDetails> extensions = new HashSet<ExtensionDetails>();
        ExtensionUtils.findExtensionsInZips(archiveFiles, extensions);
        ExtensionUtils.findExtensionsInFolder(archiveDir.getFile(false), extensions);
        return extensions;
    }

    private static void findExtensionsInZips(ResourceFile[] archiveFiles, Set<ExtensionDetails> extensions) {
        for (ResourceFile file : archiveFiles) {
            ExtensionDetails extension = ExtensionUtils.createExtensionDetailsFromArchive(file);
            if (extension == null) {
                log.trace("Skipping archive file; not an extension: " + file);
                continue;
            }
            if (extensions.contains(extension)) {
                log.error("Skipping extension \"" + extension.getName() + "\" found at " + extension.getInstallPath() + ".\nArchived extension by that name already found.");
            }
            extensions.add(extension);
        }
    }

    private static void findExtensionsInFolder(File dir, Set<ExtensionDetails> extensions) {
        List<File> propFiles = ExtensionUtils.findExtensionPropertyFiles(dir);
        for (File propFile : propFiles) {
            ExtensionDetails extension = ExtensionUtils.createExtensionFromProperties(propFile);
            if (extension == null) continue;
            File extDir = propFile.getParentFile();
            extension.setArchivePath(extDir.getAbsolutePath());
            if (extensions.contains(extension)) {
                log.error("Skipping duplicate extension \"" + extension.getName() + "\" found at " + extension.getInstallPath());
            }
            extensions.add(extension);
        }
    }

    public static boolean install(File file) {
        log.trace("Installing extension file " + file);
        if (file == null) {
            log.error("Install file cannot be null");
            return false;
        }
        ExtensionDetails extension = ExtensionUtils.getExtension(file, false);
        if (extension == null) {
            Msg.showError(ExtensionUtils.class, null, (String)"Error Installing Extension", (Object)(file.getAbsolutePath() + " does not point to a valid ghidra extension"));
            return false;
        }
        Extensions extensions = ExtensionUtils.getAllInstalledExtensions();
        if (ExtensionUtils.checkForConflictWithDevelopmentExtension(extension, extensions)) {
            return false;
        }
        if (ExtensionUtils.checkForDuplicateExtensions(extension, extensions)) {
            return false;
        }
        if (!ExtensionUtils.validateExtensionVersion(extension)) {
            return false;
        }
        AtomicBoolean installed = new AtomicBoolean(false);
        TaskLauncher.launchModal((String)"Installing Extension", monitor -> installed.set(ExtensionUtils.doRunInstallTask(extension, file, monitor)));
        boolean success = installed.get();
        if (success) {
            log.trace("Finished installing " + file);
        } else {
            log.trace("Failed to install " + file);
        }
        return success;
    }

    private static boolean doRunInstallTask(ExtensionDetails extension, File file, TaskMonitor monitor) {
        try {
            if (file.isFile()) {
                return ExtensionUtils.unzipToInstallationFolder(extension, file, monitor);
            }
            return ExtensionUtils.copyToInstallationFolder(file, monitor);
        }
        catch (CancelledException e) {
            log.info("Extension installation cancelled by user");
        }
        catch (IOException e) {
            Msg.showError(ExtensionUtils.class, null, (String)"Error Installing Extension", (Object)"Unexpected error installing extension", (Throwable)e);
        }
        return false;
    }

    public static boolean installExtensionFromArchive(ExtensionDetails extension) {
        if (extension == null) {
            log.error("Extension to install cannot be null");
            return false;
        }
        String archivePath = extension.getArchivePath();
        if (archivePath == null) {
            log.error("Cannot install from archive; extension is missing archive path");
            return false;
        }
        ApplicationLayout layout = Application.getApplicationLayout();
        ResourceFile extInstallDir = (ResourceFile)layout.getExtensionInstallationDirs().get(0);
        String extName = extension.getName();
        File extDestinationDir = new ResourceFile(extInstallDir, extName).getFile(false);
        File archiveFile = new File(archivePath);
        if (ExtensionUtils.install(archiveFile)) {
            extension.setInstallDir(new File(extDestinationDir, extName));
            return true;
        }
        return false;
    }

    private static boolean validateExtensionVersion(ExtensionDetails extension) {
        String appVersion;
        String extVersion = extension.getVersion();
        if (extVersion == null) {
            extVersion = "<no version>";
        }
        if (extVersion.equals(appVersion = Application.getApplicationVersion())) {
            return true;
        }
        String message = "Extension version mismatch.\nName: " + extension.getName() + "Extension version: " + extVersion + ".\nGhidra version: " + appVersion + ".";
        int choice = OptionDialog.showOptionDialogWithCancelAsDefaultButton(null, (String)"Extension Version Mismatch", (String)message, (String)"Install Anyway");
        if (choice != 1) {
            log.info(ExtensionUtils.removeNewlines(message + " Did not install"));
            return false;
        }
        return true;
    }

    private static String removeNewlines(String s) {
        return s.replaceAll("\n", " ");
    }

    private static boolean checkForDuplicateExtensions(ExtensionDetails newExtension, Extensions extensions) {
        String name = newExtension.getName();
        log.trace("Checking for duplicate extensions for '" + name + "'");
        List<ExtensionDetails> matches = extensions.getMatchingExtensions(newExtension);
        if (matches.isEmpty()) {
            log.trace("No matching extensions installed");
            return false;
        }
        log.trace("Duplicate extensions found by name '" + name + "'");
        if (matches.size() > 1) {
            ExtensionUtils.reportMultipleDuplicateExtensionsWhenInstalling(newExtension, matches);
            return true;
        }
        ExtensionDetails installedExtension = matches.get(0);
        String message = "Attempting to install an extension matching the name of an existing extension.\nNew extension version: " + newExtension.getVersion() + ".\nInstalled extension version: " + installedExtension.getVersion() + ".\n\nTo install, click 'Remove Existing', restart Ghidra, then install again.";
        int choice = OptionDialog.showOptionDialogWithCancelAsDefaultButton(null, (String)"Duplicate Extension", (String)message, (String)"Remove Existing");
        String installPath = installedExtension.getInstallPath();
        if (choice != 1) {
            log.info(ExtensionUtils.removeNewlines(message + " Skipping installation. Original extension still installed: " + installPath));
            return true;
        }
        log.info(ExtensionUtils.removeNewlines(message + " Installing new extension. Existing extension will be removed after restart: " + installPath));
        installedExtension.markForUninstall();
        return true;
    }

    private static void reportMultipleDuplicateExtensionsWhenInstalling(ExtensionDetails extension, List<ExtensionDetails> matches) {
        StringBuilder buffy = new StringBuilder();
        buffy.append("Found multiple duplicate extensions while trying to install '").append(extension.getName()).append("'\n");
        for (ExtensionDetails otherExtension : matches) {
            buffy.append("Duplicate: " + otherExtension.getInstallPath()).append('\n');
        }
        buffy.append("Please close Ghidra and manually remove from these extensions from the filesystem.");
        Msg.showInfo(ExtensionUtils.class, null, (String)"Duplicate Extensions Found", (Object)buffy.toString());
    }

    private static void reportDuplicateExtensionsWhenLoading(String name, List<ExtensionDetails> extensions) {
        ExtensionDetails loadedExtension = extensions.get(0);
        File loadedInstallDir = loadedExtension.getInstallDir();
        for (int i = 1; i < extensions.size(); ++i) {
            ExtensionDetails duplicate = extensions.get(i);
            log.info("Duplicate extension found '" + name + "'.  Keeping extension from " + loadedInstallDir + ".  Skipping extension found at " + duplicate.getInstallDir());
        }
    }

    private static boolean checkForConflictWithDevelopmentExtension(ExtensionDetails newExtension, Extensions extensions) {
        String name = newExtension.getName();
        log.trace("Checking for duplicate dev mode extensions for '" + name + "'");
        List<ExtensionDetails> matches = extensions.getMatchingExtensions(newExtension);
        if (matches.isEmpty()) {
            log.trace("No matching extensions installed");
            return false;
        }
        for (ExtensionDetails extension : matches) {
            if (!extension.isInstalledInInstallationFolder()) continue;
            String message = "Attempting to install an extension that conflicts with an extension located in the Ghidra installation folder.\nYou must manually remove the existing extension to install the new extension.\nExisting extension: " + extension.getInstallDir();
            log.trace(ExtensionUtils.removeNewlines(message));
            OkDialog.showError((String)"Duplicate Extensions Found", (String)message);
            return true;
        }
        return false;
    }

    public static boolean isExtension(File file) {
        return ExtensionUtils.getExtension(file, true) != null;
    }

    private static ExtensionDetails getExtension(File file, boolean quiet) {
        if (file == null) {
            log.error("Cannot get an extension; null file");
            return null;
        }
        try {
            return ExtensionUtils.tryToGetExtension(file);
        }
        catch (IOException e) {
            if (quiet) {
                log.trace("Exception trying to read an extension from " + file, (Throwable)e);
            } else {
                log.error("Exception trying to read an extension from " + file, (Throwable)e);
            }
            return null;
        }
    }

    private static ExtensionDetails tryToGetExtension(File file) throws IOException {
        File propertyFile;
        if (file == null) {
            log.error("Cannot get an extension; null file");
            return null;
        }
        if (file.isDirectory() && file.canRead() && (propertyFile = new File(file, PROPERTIES_FILE_NAME)).isFile()) {
            return ExtensionUtils.tryToLoadExtensionFromProperties(propertyFile);
        }
        if (ExtensionUtils.isZip(file)) {
            try (ZipFile zipFile = new ZipFile(file);){
                Properties props = ExtensionUtils.getProperties(zipFile);
                if (props != null) {
                    ExtensionDetails extensionDetails = ExtensionUtils.createExtensionDetails(props);
                    return extensionDetails;
                }
                throw new IOException("No extension.properties file found in zip");
            }
        }
        return null;
    }

    private static boolean isZip(File file) {
        boolean bl;
        if (file == null) {
            log.error("Cannot check for extension zip; null file");
            return false;
        }
        if (file.isDirectory()) {
            return false;
        }
        if (file.length() < 4L) {
            return false;
        }
        DataInputStream in = new DataInputStream(new BufferedInputStream(new FileInputStream(file)));
        try {
            int test = in.readInt();
            bl = test == 1347093252;
        }
        catch (Throwable throwable) {
            try {
                try {
                    in.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                log.trace("Unable to check if file is a zip file: " + file + ". " + e.getMessage());
                return false;
            }
        }
        in.close();
        return bl;
    }

    private static List<File> findExtensionPropertyFiles(File installDir) {
        ArrayList<File> results = new ArrayList<File>();
        FileUtilities.forEachFile((File)installDir, f -> {
            if (!f.isDirectory() || f.getName().equals("Skeleton")) {
                return;
            }
            File pf = ExtensionUtils.getPropertyFile(f);
            if (pf != null) {
                results.add(pf);
            }
        });
        return results;
    }

    private static File getPropertyFile(File dir) {
        File f = new File(dir, PROPERTIES_FILE_NAME_UNINSTALLED);
        if (f.exists()) {
            return f;
        }
        f = new File(dir, PROPERTIES_FILE_NAME);
        if (f.exists()) {
            return f;
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static ExtensionDetails createExtensionDetailsFromArchive(ResourceFile resourceFile) {
        File file = resourceFile.getFile(false);
        if (!ExtensionUtils.isZip(file)) {
            return null;
        }
        try (ZipFile zipFile = new ZipFile(file);){
            Properties props = ExtensionUtils.getProperties(zipFile);
            if (props == null) return null;
            ExtensionDetails extension = ExtensionUtils.createExtensionDetails(props);
            extension.setArchivePath(file.getAbsolutePath());
            ExtensionDetails extensionDetails = extension;
            return extensionDetails;
        }
        catch (IOException e) {
            log.error("Unable to read zip file to get extension properties: " + file, (Throwable)e);
        }
        return null;
    }

    private static Properties getProperties(ZipFile zipFile) throws IOException {
        Properties props = null;
        Enumeration zipEntries = zipFile.getEntries();
        while (zipEntries.hasMoreElements()) {
            ZipArchiveEntry entry = (ZipArchiveEntry)zipEntries.nextElement();
            Properties nextProperties = ExtensionUtils.getProperties(zipFile, entry);
            if (nextProperties == null) continue;
            if (props != null) {
                throw new IOException("Zip file contains multiple extension properties files");
            }
            props = nextProperties;
        }
        return props;
    }

    private static Properties getProperties(ZipFile zipFile, ZipArchiveEntry entry) throws IOException {
        String path = entry.getName();
        List parts = FileUtilities.pathToParts((String)path);
        if (parts.size() != 2) {
            return null;
        }
        if (!entry.getName().endsWith(PROPERTIES_FILE_NAME)) {
            return null;
        }
        InputStream propFile = zipFile.getInputStream(entry);
        Properties prop = new Properties();
        prop.load(propFile);
        return prop;
    }

    private static boolean copyToInstallationFolder(File sourceFolder, TaskMonitor monitor) throws IOException, CancelledException {
        log.trace("Copying extension from " + sourceFolder);
        ApplicationLayout layout = Application.getApplicationLayout();
        ResourceFile installDir = (ResourceFile)layout.getExtensionInstallationDirs().get(0);
        File installDirRoot = installDir.getFile(false);
        File newDir = new File(installDirRoot, sourceFolder.getName());
        if (ExtensionUtils.hasExistingExtension(newDir, monitor)) {
            return false;
        }
        log.trace("Copying extension to " + newDir);
        FileUtilities.copyDir((File)sourceFolder, (File)newDir, (TaskMonitor)monitor);
        return true;
    }

    private static boolean hasExistingExtension(File extensionFolder, TaskMonitor monitor) {
        if (extensionFolder.exists()) {
            Msg.showWarn(ExtensionUtils.class, null, (String)"Duplicate Extension Folder", (Object)("Attempting to install a new extension over an existing directory.\nEither remove the extension for that directory from the UI\nor close Ghidra and delete the directory and try installing again.\n\nDirectory: " + extensionFolder));
            return true;
        }
        return false;
    }

    private static boolean unzipToInstallationFolder(ExtensionDetails extension, File file, TaskMonitor monitor) throws CancelledException, IOException {
        log.trace("Unzipping extension from " + file);
        ApplicationLayout layout = Application.getApplicationLayout();
        ResourceFile installDir = (ResourceFile)layout.getExtensionInstallationDirs().get(0);
        File installDirRoot = installDir.getFile(false);
        File destinationFolder = new File(installDirRoot, extension.getName());
        if (ExtensionUtils.hasExistingExtension(destinationFolder, monitor)) {
            return false;
        }
        try (ZipFile zipFile = new ZipFile(file);){
            Enumeration entries = zipFile.getEntries();
            while (entries.hasMoreElements()) {
                monitor.checkCancelled();
                ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
                String filePath = installDir + File.separator + entry.getName();
                File destination = new File(filePath);
                if (entry.isDirectory()) {
                    destination.mkdirs();
                    continue;
                }
                ExtensionUtils.writeZipEntryToFile(zipFile, entry, destination);
            }
        }
        return true;
    }

    private static void writeZipEntryToFile(ZipFile zFile, ZipArchiveEntry entry, File destination) throws IOException {
        try (BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(destination));){
            IOUtils.copy((InputStream)zFile.getInputStream(entry), (OutputStream)outputStream);
            if (entry.getPlatform() != 3) {
                return;
            }
            int mode = entry.getUnixMode();
            if (mode != 0) {
                Set<PosixFilePermission> perms = ExtensionUtils.getPermissions(mode);
                try {
                    Files.setPosixFilePermissions(destination.toPath(), perms);
                }
                catch (UnsupportedOperationException unsupportedOperationException) {
                    // empty catch block
                }
            }
        }
    }

    private static ExtensionDetails tryToLoadExtensionFromProperties(File file) throws IOException {
        Properties props = new Properties();
        try (FileInputStream in = new FileInputStream(file.getAbsolutePath());){
            props.load(in);
            ExtensionDetails extensionDetails = ExtensionUtils.createExtensionDetails(props);
            return extensionDetails;
        }
    }

    private static ExtensionDetails createExtensionFromProperties(File file) {
        try {
            return ExtensionUtils.tryToLoadExtensionFromProperties(file);
        }
        catch (IOException e) {
            log.error("Error loading extension properties from " + file.getAbsolutePath(), (Throwable)e);
            return null;
        }
    }

    private static ExtensionDetails createExtensionDetails(Properties props) {
        String name = props.getProperty("name");
        String desc = props.getProperty("description");
        String author = props.getProperty("author");
        String date = props.getProperty("createdOn");
        String version = props.getProperty("version");
        return new ExtensionDetails(name, desc, author, date, version);
    }

    private static boolean removeExtension(ExtensionDetails extension) {
        if (extension == null) {
            log.error("Extension to uninstall cannot be null");
            return false;
        }
        File installDir = extension.getInstallDir();
        if (installDir == null) {
            log.error("Extension installation path is not set; unable to delete files");
            return false;
        }
        if (FileUtilities.deleteDir((File)installDir)) {
            extension.setInstallDir(null);
            return true;
        }
        return false;
    }

    private static Set<PosixFilePermission> getPermissions(int unixMode) {
        HashSet<PosixFilePermission> permissions = new HashSet<PosixFilePermission>();
        if ((unixMode & 0x100) != 0) {
            permissions.add(PosixFilePermission.OWNER_READ);
        }
        if ((unixMode & 0x80) != 0) {
            permissions.add(PosixFilePermission.OWNER_WRITE);
        }
        if ((unixMode & 0x40) != 0) {
            permissions.add(PosixFilePermission.OWNER_EXECUTE);
        }
        if ((unixMode & 0x20) != 0) {
            permissions.add(PosixFilePermission.GROUP_READ);
        }
        if ((unixMode & 0x10) != 0) {
            permissions.add(PosixFilePermission.GROUP_WRITE);
        }
        if ((unixMode & 8) != 0) {
            permissions.add(PosixFilePermission.GROUP_EXECUTE);
        }
        if ((unixMode & 4) != 0) {
            permissions.add(PosixFilePermission.OTHERS_READ);
        }
        if ((unixMode & 2) != 0) {
            permissions.add(PosixFilePermission.OTHERS_WRITE);
        }
        if ((unixMode & 1) != 0) {
            permissions.add(PosixFilePermission.OTHERS_EXECUTE);
        }
        return permissions;
    }

    private static class Extensions {
        private Map<String, List<ExtensionDetails>> extensionsByName = new HashMap<String, List<ExtensionDetails>>();

        private Extensions() {
        }

        void add(ExtensionDetails e) {
            this.extensionsByName.computeIfAbsent(e.getName(), n -> new ArrayList()).add(e);
        }

        Set<ExtensionDetails> getActiveExtensions() {
            return this.extensionsByName.values().stream().map(list -> (ExtensionDetails)list.get(0)).filter(ext -> !ext.isPendingUninstall()).collect(Collectors.toSet());
        }

        List<ExtensionDetails> getMatchingExtensions(ExtensionDetails extension) {
            return this.extensionsByName.computeIfAbsent(extension.getName(), name -> List.of());
        }

        void cleanupExtensionsMarkedForRemoval() {
            HashSet<String> names = new HashSet<String>(this.extensionsByName.keySet());
            for (String name : names) {
                List<ExtensionDetails> extensions = this.extensionsByName.get(name);
                Iterator<ExtensionDetails> it = extensions.iterator();
                while (it.hasNext()) {
                    ExtensionDetails extension = it.next();
                    if (!extension.isPendingUninstall()) continue;
                    if (!ExtensionUtils.removeExtension(extension)) {
                        log.error("Error removing extension: " + extension.getInstallPath());
                    }
                    it.remove();
                }
                if (!extensions.isEmpty()) continue;
                this.extensionsByName.remove(name);
            }
        }

        void reportDuplicateExtensions() {
            Set<Map.Entry<String, List<ExtensionDetails>>> entries = this.extensionsByName.entrySet();
            for (Map.Entry<String, List<ExtensionDetails>> entry : entries) {
                List<ExtensionDetails> list = entry.getValue();
                if (list.size() == 1) continue;
                ExtensionUtils.reportDuplicateExtensionsWhenLoading(entry.getKey(), list);
            }
        }

        Set<ExtensionDetails> get() {
            return this.extensionsByName.values().stream().map(list -> (ExtensionDetails)list.get(0)).collect(Collectors.toSet());
        }

        String getAsString() {
            StringBuilder buffy = new StringBuilder();
            Set<Map.Entry<String, List<ExtensionDetails>>> entries = this.extensionsByName.entrySet();
            for (Map.Entry<String, List<ExtensionDetails>> entry : entries) {
                String name = entry.getKey();
                buffy.append("Name: ").append(name);
                List<ExtensionDetails> extensions = entry.getValue();
                if (extensions.size() == 1) {
                    buffy.append(" - ").append(extensions.get(0).getInstallDir()).append('\n');
                    continue;
                }
                for (ExtensionDetails e : extensions) {
                    buffy.append("\t").append(e.getInstallDir()).append('\n');
                }
            }
            if (buffy.isEmpty()) {
                return "<no extensions installed>";
            }
            if (!buffy.isEmpty()) {
                buffy.deleteCharAt(buffy.length() - 1);
            }
            return buffy.toString();
        }
    }
}

