// (c) Microsoft Corporation 2005-2007.  

#light

namespace Microsoft.FSharp.Compatibility.FSharp

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Control.Lazy
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Control
open Microsoft.FSharp.Collections
open System
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

type 'a LazyList =
    { mutable status : LazyCellStatus< 'a > }
    
    member x.Force = 
        match x.status with 
        | LazyCellStatus.Delayed f -> 
            x.status <- Exception Undefined; 
            begin 
              try let res = f () in x.status <- LazyCellStatus.Value res; res 
              with e -> x.status <- LazyCellStatus.Exception(e); rethrow()
            end
        | LazyCellStatus.Value x -> x
        | LazyCellStatus.Exception e -> raise e
    
    member s.GetEnumeratorImpl() = 
            let getcell (x : 'a LazyList) = x.Force
            let to_seq s = Seq.unfold (fun ll -> match getcell ll with CellNil -> None | CellCons(a,b) -> Some(a,b)) s 
            (to_seq s).GetEnumerator()
            
    interface IEnumerable<'a> with
        member s.GetEnumerator() = s.GetEnumeratorImpl()

    interface System.Collections.IEnumerable with
        override s.GetEnumerator() = (s.GetEnumeratorImpl() :> System.Collections.IEnumerator)

and LazyCellStatus<'a> =
    | Delayed of (unit -> LazyListCell <'a> )
    | Value of LazyListCell <'a> 
    | Exception of System.Exception

and LazyListCell<'a> = 
    | CellCons of 'a * 'a LazyList 
    | CellNil

[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module LazyList = 

    let lzy f = { status = Delayed f }
    let force (x: 'a LazyList) = x.Force

    let notlazy v = { status = Value v }
    
    type 'a t = 'a LazyList
    type 'a llist = 'a LazyList

    type LazyItem<'a> = Cons of 'a * 'a LazyList | Nil
    type 'a item = 'a LazyItem
    let get (x : 'a LazyList) = match force x with CellCons (a,b) -> Some(a,b) | CellNil -> None
    let getcell (x : 'a LazyList) = force x 
    let empty () = notlazy CellNil
    let make x = (x : 'a LazyList)
    let consc x l = CellCons(x,l)
    let cons x l = lzy(fun () -> (consc x l))
    let consf x l = lzy(fun () -> (consc x (lzy(fun () ->  (force (l()))))))

    let rec unfold f z = 
      lzy(fun () -> 
          match f z with
          | None       -> CellNil
          | Some (x,z) -> CellCons (x,unfold f z))

    let rec append l1  l2 = lzy(fun () ->  (appendc l1 l2))
    and appendc l1 l2 =
      match getcell l1 with
      | CellNil -> force l2
      | CellCons(a,b) -> consc a (append b l2)

    let delayed f = lzy(fun () ->  (getcell (f())))
    let repeat x = 
      let rec s = cons x (delayed (fun () -> s)) in s

    let rec map f s = 
      lzy(fun () ->  
        match getcell s with
        | CellNil -> CellNil
        | CellCons(a,b) -> consc (f a) (map f b))

    let rec map2 f s1 s2 =  
      lzy(fun () -> 
        match getcell s1, getcell s2  with
        | CellCons(a1,b1),CellCons(a2,b2) -> consc (f a1 a2) (map2 f b1 b2)
        | _ -> CellNil)

    let rec combine s1 s2 = 
      lzy(fun () -> 
        match getcell s1, getcell s2  with
        | CellCons(a1,b1),CellCons(a2,b2) -> consc (a1,a2) (combine b1 b2)
        | _ -> CellNil)

    let rec concat s1 = 
      lzy(fun () -> 
        match getcell s1 with
        | CellCons(a,b) -> appendc a (concat b)
        | CellNil -> CellNil)
      
    let rec filter p s1= lzy(fun () ->  filterc p s1)
    and filterc p s1 =
        match getcell s1 with
        | CellCons(a,b) -> if p a then consc a (filter p b) else filterc p b
        | CellNil -> CellNil
      
    let rec first p s1 =
        match getcell s1 with
        | CellCons(a,b) -> if p a then Some a else first p b
        | CellNil -> None

    let find p s1 =
        match first p s1 with
        | Some a -> a
        | None   -> not_found()

    let find_all p s1= filter p s1 (* deprecated *)
    let flatten s1= concat s1      (* deprecated *)

    let rec folds f acc s1 = 
      lzy(fun () -> 
        match getcell s1 with
        | CellCons(a,b) -> let acc' = f acc a in consc acc' (folds f acc' b)
        | CellNil -> CellNil)

    let hd s1 = 
      match getcell s1 with
      | CellCons(a,b) -> a
      | CellNil -> invalid_arg "hd"

    let tl s1 = 
      match getcell s1 with
      | CellCons(a,b) -> b
      | CellNil -> invalid_arg "tl"

    let nonempty s1 =
      match getcell s1 with
      | CellCons(a,b) -> true
      | CellNil -> false

    let rec take n s = 
      lzy(fun () -> 
        if n < 0 then invalid_arg "take: -ve number" 
        elif n = 0 then CellNil 
        else
          match getcell s with
          | CellCons(a,s) -> consc a (take (n-1) s)
          | CellNil -> invalid_arg "take: not enough items in lzy(fun () ->  list" )

    let rec drop n s = 
      lzy(fun () -> 
        if n < 0 then invalid_arg "drop: -ve number" 
        else dropc n s)
    and dropc n s =
      if n = 0 then force s 
      else  
        match getcell s with
        | CellCons(a,s) -> dropc (n-1) s
        | CellNil -> invalid_arg "drop: not enough items in lazy list"

    let rec of_list l = 
      lzy(fun () -> 
        match l with [] -> CellNil | h :: t -> consc h (of_list t))
      
    let rec to_list s = 
      match getcell s with
      | CellNil -> []
      | CellCons(h,t) -> h :: to_list t
      
    let rec copy_from i a = 
      lzy(fun () -> 
        if i >= Array.length a then CellNil 
        else consc a.(i) (copy_from (i+1) a))
      
    let rec copy_to arr s i = 
      match getcell s with
      | CellNil -> ()
      | CellCons(a,b) -> arr.(i) <- a; copy_to arr b (i+1)

      
    let of_array a = copy_from 0 a
    let to_array s = Array.of_list (to_list s)
      
    let rec mem x s = 
      match getcell s with
      | CellNil -> false
      | CellCons(a,b) -> x = a || mem x b

    let rec length_aux n s = 
      match getcell s with
      | CellNil -> 0
      | CellCons(_,b) -> length_aux (n+1) b

    let length s = length_aux 0 s

    let to_seq (s: LazyList<'a>) = (s :> IEnumerable<_>)
    let to_IEnumerable s = to_seq s

    #if CLI_AT_MOST_1_1
    #else
    let to_ICollection s = 
      { new ICollection<'a> 
          with Add(x) = raise (new System.NotSupportedException("ReadOnlyCollection"));
          and Clear() = raise (new System.NotSupportedException("ReadOnlyCollection"));
          and Remove(x) = raise (new System.NotSupportedException("ReadOnlyCollection"));
          and Contains(x) = mem x s
          and CopyTo(arr,i) = copy_to arr s i
          and get_IsReadOnly() = true
          and get_Count() = length s 
        interface IEnumerable<'a>
          with GetEnumerator() = (to_seq s).GetEnumerator()
        interface System.Collections.IEnumerable
          with GetEnumerator() = ((to_seq s).GetEnumerator() :> System.Collections.IEnumerator)
        }
      
    #endif


    let rec of_fresh_IEnumerator (e : IEnumerator<_>) = 
      lzy(fun () -> 
        if e.MoveNext() then 
          consc e.Current (of_fresh_IEnumerator e)
        else CellNil)
      
    let of_seq (c :> IEnumerable<_>) =
      of_fresh_IEnumerator (c.GetEnumerator()) 
      
    let of_IEnumerable c = of_seq c
      
#if CLI_AT_MOST_1_1
#else
    let of_ICollection (c : 'd :> ICollection<'a>) : LazyList<'a> =
      of_fresh_IEnumerator ((c :> IEnumerable<'a>).GetEnumerator()) 

#endif

    let (|Cons|Nil|) l = match getcell l with CellCons(a,b) -> Cons(a,b) | CellNil -> Nil

