// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.Text.StructuredFormat

open System
open System.IO
open Microsoft.FSharp.Core
open Microsoft.FSharp.Text
open Microsoft.FSharp.Collections

/// A record of options to control structural formatting.
/// For F# Interactive properties matching those of this value can be accessed via the 'fsi'
/// value.
/// 
/// Floating Point format given in the same format accepted by System.Double.ToString,
/// e.g. f6 or g15.
///
/// If ShowProperties is set the printing process will evaluate properties of the values being
/// displayed.  This may cause additional computation.  
///
/// The ShowIEnumerable is set the printing process will force the evalution of IEnumerable objects
/// to a small, finite depth, as determined by the printing parameters.
/// This may lead to additional computation being performed during printing.
///
/// <example>
/// From F# Interactive the default settings can be adjusted using, for example, 
/// <pre>
///   open Microsoft.FSharp.Compiler.Interactive.Settings;;
///   setPrintWidth 120;;
/// </pre>
/// </example>
type FormatOptions 
  = { FloatingPointFormat: string
      AttributeProcessor: (string -> (string * string) list -> bool -> unit)
      PrintIntercepts:  (StructuredFormat.IEnvironment -> obj -> StructuredFormat.Layout option) list
      FormatProvider: System.IFormatProvider
      PrintWidth : int 
      PrintDepth : int 
      PrintLength : int
      ShowProperties : bool
      ShowIEnumerable: bool  }
 with 
   static member Default : FormatOptions
 end

module Display = 


    /// Convert any value to a string using a standard formatter
    /// Data is typically formatted in a structured format, e.g.
    /// lists are formatted using the "[1;2]" notation.
    /// The details of the format are not specified and may change
    /// from version to version and according to the flags given
    /// to the F# compiler.  The format is intended to be human-readable,
    /// not machine readable.  If alternative generic formats are required
    /// you should develop your own formatter, using the code in the
    /// implementation of this file as a starting point.
    ///
    /// Data from other .NET languages is formatted using a virtual
    /// call to Object.ToString() on the boxed version of the input.
    val any_to_string: 'a -> string

    /// Ouput any value to a channel using the same set of formatting rules
    /// as any_to_string
    val output_any: #TextWriter -> 'a -> unit

    /// Convert any value to a layout using the given formatting options.  The
    /// layout can then be processed using formatting display engines such as
    /// those in the LayoutOps module.  any_to_string and output_any are
    /// built using any_to_layout with default format options.
    val any_to_layout   : FormatOptions -> 'a -> StructuredFormat.Layout
    val squash_layout   : FormatOptions -> StructuredFormat.Layout -> StructuredFormat.Layout
    val output_layout   : FormatOptions -> #TextWriter -> StructuredFormat.Layout -> unit
    val layout_to_string: FormatOptions -> StructuredFormat.Layout -> string
    val layout_as_string: FormatOptions -> 'a -> string
