// The purpose of this file is to connect the Caml and F# code in ilwrite.ml
// to the strong-name signer API provided by the CLR.  The code is used by all
// versions of the Abstract IL library and F# compiler that support strong
// name signing.  There is no standard managed API for strong name signing short of
// using Reflection.Emit (which we may use in the future)

#include "common.h"

#include <mscoree.h>
#include "cordefs.h"
#include <StrongName.h>

static HINSTANCE g_SnDll = NULL;
static HRESULT InitSnDll(bool forceUseOfSSCLI)
{
   if (!g_SnDll)
   {
#if !(defined FEATURE_PAL) && !(defined PLATFORM_UNIX)
       char *dll = forceUseOfSSCLI ? "sscoree.dll" : MSCOREE_SHIM_A;
#else
       char *dll = MSCOREE_SHIM_A;
#endif
       g_SnDll = LoadLibraryA(dll);
       if (!g_SnDll)
       {
           //printf("Failed to dynamically link to %s\n", dll);
           return E_FAIL;
       }
   }
    return S_OK;
}


#define SNCALL(res, nm, argtys, args) \
 { typedef SNAPI Proc argtys; \
   Proc *p = p = (Proc *) GetProcAddress(g_SnDll, #nm ); \
   if (!p) \
	   return E_FAIL; \
   res = (*p) args; }

//-----------------------------------------------------------------------------
// Entry points used by both the Augmented-Managed (i.e. F#-on-windows) and 
// OCaml compilations of this code.
//-----------------------------------------------------------------------------


EXTAPI signerReadBinaryFile(bool sscli, WCHAR *file, BYTE *pPublicKey, ULONG *pPublicKeySize)
{
    IfFailRet(InitSnDll(sscli),"loading strong name signing component");

    // Read key pair from file.
    HANDLE hFile = CreateFileW(file,
	GENERIC_READ,
	FILE_SHARE_READ,
	NULL,
	OPEN_EXISTING,
	FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN,
	NULL);
    if(hFile == INVALID_HANDLE_VALUE)
	return E_FAIL;

    ULONG publicKeySize = GetFileSize(hFile, NULL);
    if (pPublicKeySize)
	*pPublicKeySize = publicKeySize;

    if (pPublicKey)
    {
	DWORD dwBytesRead;
	if (!ReadFile(hFile, pPublicKey, publicKeySize, &dwBytesRead, NULL))
	    return E_FAIL;
    }
    CloseHandle(hFile);
    return S_OK;
}


EXTAPI signerGetPublicKeyForKeyPair(bool sscli,  BYTE *pKeyPair, ULONG keyPairSize, BYTE *pPublicKey, ULONG *pPublicKeySize)
{

    BYTE *publicKeyInternal;
    BOOL snres;
    SNCALL(snres,StrongNameGetPublicKey,(LPCWSTR,BYTE *, ULONG, BYTE **, ULONG *),
	(NULL,pKeyPair,keyPairSize,&publicKeyInternal,pPublicKeySize));

    if (snres && pPublicKey)
    {
	memcpy(pPublicKey,publicKeyInternal,*pPublicKeySize);
    }
    //BOOL snres2;
    //SNCALL(snres2,StrongNameFreeBuffer,(BYTE *),(publicKeyInternal));

    return snres ? S_OK : E_FAIL;

}


EXTAPI signerGetPublicKeyForKeyContainer(bool sscli, WCHAR *keyContainerName, BYTE *pPublicKey, ULONG *pPublicKeySize)
{
    BYTE *publicKeyInternal;
    BOOL snres;
    SNCALL(snres, StrongNameGetPublicKey, (LPCWSTR,BYTE *, ULONG, BYTE **, ULONG *),
	(keyContainerName,NULL,0,(pPublicKey ? &publicKeyInternal : NULL),pPublicKeySize));
    if (snres && pPublicKey)
    {
	memcpy(pPublicKey,publicKeyInternal,*pPublicKeySize);
    }
    BOOL snres2;
    SNCALL(snres2,StrongNameFreeBuffer,(BYTE *),(publicKeyInternal));
    return snres ? S_OK : E_FAIL;
}



// Delete any stale container.
EXTAPI signerCloseKeyContainer(bool sscli, WCHAR *keyContainerName)
{
    IfFailRet(InitSnDll(sscli),"loading strong name signing component");
    BOOL snres;
    SNCALL(snres, StrongNameKeyDelete, (LPCWSTR), (keyContainerName));
    return snres ? S_OK : E_FAIL;
}

// Determine size of signature blob.
EXTAPI signerSignatureSize(bool sscli, BYTE *pPublicKey, ULONG publicKeySize, DWORD *res)
{
    IfFailRet(InitSnDll(sscli),"loading strong name signing component");
    BOOL snres;
    SNCALL(snres, StrongNameSignatureSize, (BYTE *, ULONG, DWORD *),
	(pPublicKey, publicKeySize, res));
    return snres ? S_OK : E_FAIL;
}
// Update the output PE image with a strong name signature.
EXTAPI signerSignFileWithKeyPair(bool sscli, LPCTSTR fileName, BYTE *pKeyPair, ULONG keyPairSize)
{
    IfFailRet(InitSnDll(sscli),"loading strong name signing component");

    BOOL snres;
    SNCALL(snres, StrongNameSignatureGeneration,
	(LPCWSTR, LPCWSTR, BYTE *, ULONG, BYTE **, ULONG *),
	(fileName,NULL,pKeyPair,keyPairSize,NULL,NULL));
    if (!snres)
	return E_FAIL;

    BOOLEAN ok;

    SNCALL(snres, StrongNameSignatureVerificationEx,
	(LPCWSTR, BOOLEAN, BOOLEAN    *),
	(fileName,TRUE,&ok));
    return snres ? S_OK : E_FAIL;

}

// Update the output PE image with a strong name signature.
EXTAPI signerSignFileWithKeyContainer(bool sscli, LPCTSTR fileName, LPCTSTR keyContainerName)
{
    BOOL snres;
    SNCALL(snres, StrongNameSignatureGeneration,
	(LPCWSTR, LPCWSTR, BYTE *, ULONG, BYTE **, ULONG *),
	(fileName,keyContainerName,NULL,NULL,NULL,NULL));
    if (!snres)
	return E_FAIL;

    BOOLEAN ok;

    SNCALL(snres, StrongNameSignatureVerificationEx,
	(LPCWSTR, BOOLEAN, BOOLEAN    *),
	(fileName,TRUE,&ok));
    return snres ? S_OK : E_FAIL;

}




#ifdef CAML_STUBS

//-----------------------------------------------------------------------------
// OCaml stubs for the above
//-----------------------------------------------------------------------------

extern "C"
CAMLprim value signerOpenPublicKeyFileCaml(value sscli, value pubkeyfile)
{
    CAMLparam2(sscli,pubkeyfile);
    
    WCHAR file[_MAX_PATH];
    
    MultiByteToWideChar(CP_ACP, 0, String_val(pubkeyfile), -1, file, _MAX_PATH);
    
    ULONG publicKeySize = 0;
    IfFailThrow(signerReadBinaryFile(Bool_val(sscli), file, NULL, &publicKeySize), "opening public key file");

    CAMLlocal1(publicKeyVal);
    publicKeyVal = alloc_string(publicKeySize);
    BYTE *publicKey = (BYTE *) String_val(publicKeyVal);

    IfFailThrow(signerReadBinaryFile(Bool_val(sscli), file, publicKey, &publicKeySize), "opening public key file");

    CAMLreturn (publicKeyVal);
}

extern "C"
CAMLprim value signerGetPublicKeyForKeyContainerCaml(value sscli, value containerName)
{
    CAMLparam2(sscli,containerName);
    WCHAR *keyContainerName = (WCHAR *) (String_val(containerName));
    
    ULONG publicKeySize = 0;
    IfFailThrow(signerGetPublicKeyForKeyContainer(Bool_val(sscli), keyContainerName, NULL, &publicKeySize), "getting publc key for key container");
    CAMLlocal1(publicKeyVal);
    publicKeyVal = alloc_string(publicKeySize);
    BYTE *publicKey = (BYTE *) String_val(publicKeyVal);
    IfFailThrow(signerGetPublicKeyForKeyContainer(Bool_val(sscli), keyContainerName, publicKey, &publicKeySize), "getting publc key for key container");

    CAMLreturn (publicKeyVal);
}

extern "C"
CAMLprim value signerGetPublicKeyForKeyPairCaml(value sscli, value keyPair)
{
    CAMLparam2(sscli,keyPair);
    
    BYTE *keyPairC = (BYTE *) (String_val(keyPair));
    ULONG keyPairSize = (string_length(keyPair));

    ULONG publicKeySize = 0;
    IfFailThrow(signerGetPublicKeyForKeyPair(Bool_val(sscli), keyPairC, keyPairSize, NULL, &publicKeySize), "getting size of public key for key pair");
    CAMLlocal1(publicKeyVal);
    publicKeyVal = alloc_string(publicKeySize);
    BYTE *publicKey = (BYTE *) String_val(publicKeyVal);
    IfFailThrow(signerGetPublicKeyForKeyPair(Bool_val(sscli), keyPairC, keyPairSize, publicKey, &publicKeySize), "getting public key for key pair");

    CAMLreturn (publicKeyVal);
}


extern "C"
CAMLprim value signerOpenKeyPairFileCaml(value sscli, value keypairfile)
{
    CAMLparam2(sscli,keypairfile);
    
   WCHAR file[_MAX_PATH];

   MultiByteToWideChar(CP_ACP, 0, String_val(keypairfile), -1, file, _MAX_PATH);

   ULONG keyPairSize = 0;
   IfFailThrow(signerReadBinaryFile(Bool_val(sscli), file, NULL, &keyPairSize), "opening key pair file");

   CAMLlocal1(keyPairVal);
   keyPairVal = alloc_string(keyPairSize);
   BYTE *keyPair = (BYTE *) String_val(keyPairVal);

   IfFailThrow(signerReadBinaryFile(Bool_val(sscli), file, keyPair, &keyPairSize), "opening key pair file");

   CAMLreturn(keyPairVal);

}


// Delete any stale container.
extern "C"
CAMLprim value signerCloseKeyContainerCaml(value sscli, value containerName)
{
    CAMLparam2(sscli,containerName);
    WCHAR *keyContainerName = (WCHAR *) (String_val(containerName));
    IfFailThrow(signerCloseKeyContainer(Bool_val(sscli), keyContainerName), "closing key container");
    CAMLreturn(Atom(0));
}

// Determine size of signature blob.
extern "C"
CAMLprim value signerSignatureSizeCaml(value sscli, value publicKey)
{
    CAMLparam2(sscli,publicKey);
    DWORD res; 
    BYTE *publicKeyC = (BYTE *) (String_val(publicKey));
    ULONG publicKeySize = string_length(publicKey);
    IfFailThrow(signerSignatureSize(Bool_val(sscli), publicKeyC, publicKeySize, &res), "determining signature size");

    CAMLreturn(Val_int(res));
}

// Update the output PE image with a strong name signature.
extern "C"
CAMLprim value signerSignFileWithKeyPairCaml(value sscli, value file, value keyPair)
{
    CAMLparam3(sscli,file,keyPair);

    WCHAR fileName[_MAX_PATH];
    
    MultiByteToWideChar(CP_ACP, 0, String_val(file), -1, fileName, _MAX_PATH);

    BYTE *keyPairC = (BYTE *) (String_val(keyPair));
    ULONG keyPairSize = (string_length(keyPair));
    IfFailThrow(signerSignFileWithKeyPair(Bool_val(sscli), fileName, keyPairC, keyPairSize), "signing and/or validating with key pair");

    CAMLreturn(Atom(0));

}

// Update the output PE image with a strong name signature.
extern "C"
CAMLprim value signerSignFileWithKeyContainerCaml(value sscli, value file, value keyContainer)
{
    CAMLparam3(sscli,file,keyContainer);

    WCHAR fileName[_MAX_PATH];
    
    MultiByteToWideChar(CP_ACP, 0, String_val(file), -1, fileName, _MAX_PATH);

    WCHAR *keyContainerName = (WCHAR *) (String_val(keyContainer));
    IfFailThrow(signerSignFileWithKeyContainer(Bool_val(sscli), fileName, keyContainerName), "signing and/or validating with key container");

    CAMLreturn(Atom(0));

}



#endif
