/**********************************************************************
Copyright (c) 2005 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.identity;

import java.util.Map;
import java.util.WeakHashMap;

import org.datanucleus.ClassConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.util.ClassUtils;

/**
 * Factory for OID instances.
 * This factory holds a cache of weak references to OID.
 *
 * @version $Revision: 1.4 $
 */
public class OIDFactory
{
    private static Map oids = new WeakHashMap();

    private OIDFactory()
    {
        // Private constructor to prevent instantiation
    }

    /**
     * Factory method for OID instances using class name and key value.
     * @param om The ObjectManager
     * @param className the pc class name
     * @param value the id value
     * @return an OID instance
     */
    public static OID getInstance(ObjectManager om, String className, Object value)
    {
        // Get the OID class to use for this ObjectManager
        Class oidClass = om.getOMFContext().getDatastoreIdentityClass();

        // Try to find a cached OID suitable
        String key = oidClass.getName() + "#" + className + "#" + value.toString();
        OID oid = (OID) oids.get(key);
        if (oid == null)
        {
            if (oidClass == ClassConstants.OID_IMPL)
            {
                //we hard code OIDImpl to improve performance
                oid = new OIDImpl(className, value);
            }
            else
            {
                //others are pluggable
                oid = (OID)ClassUtils.newInstance(oidClass, new Class[] {String.class, Object.class}, 
                    new Object[] {className, value});
            }
            oids.put(key, oid);
        }
        return oid;
    }

    /**
     * Factory method for OID instances using long key value.
     * @param om The ObjectManager
     * @param value the id value
     * @return an OID instance
     */
    public static OID getInstance(ObjectManager om, long value)
    {
        // Get the OID class to use for this ObjectManager
        Class oidClass = om.getOMFContext().getDatastoreIdentityClass();

        // Try to find a cached OID suitable
        String key = oidClass.getName() + "#" + value;
        OID oid = (OID) oids.get(key);
        if (oid == null)
        {
            if (oidClass == DatastoreUniqueOID.class)
            {
                //we hard code DatastoreUniqueOID to improve performance
                oid = new DatastoreUniqueOID(value);
            }
            else
            {
                //others are pluggable
                oid = (OID)ClassUtils.newInstance(oidClass, new Class[] {Long.class}, 
                    new Object[] {new Long(value)});
            }
            oids.put(key, oid);
        }
        return oid;
    }

    /**
     * Factory method for OID instances using toString() output.
     * @param om The ObjectManager
     * @param oidString result of toString on an OID
     * @return an OID instance
     */
    public static OID getInstance(ObjectManager om, String oidString)
    {
        // Get the OID class to use for this ObjectManager
        Class oidClass = om.getOMFContext().getDatastoreIdentityClass();

        // Try to find a cached OID suitable
        String key = oidClass.getName() + "#" + oidString;
        OID oid = (OID) oids.get(key);
        if (oid == null)
        {
            if (oidClass == ClassConstants.OID_IMPL)
            {
                //we hard code OIDImpl to improve performance
                oid = new OIDImpl(oidString);
            }
            else
            {
                //others are pluggable
                oid = (OID)ClassUtils.newInstance(oidClass, new Class[] {String.class}, new Object[] {oidString});
            }
            oids.put(key, oid);
        }
        return oid;
    }
}