/*
SRSoundManager.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRSoundManager.h"

#import "SRPrefDefaultKeys.h"

@implementation SRSoundManager

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (id)sharedInstance
{
    static SRSoundManager*  _sharedInstance = nil;
    if (!_sharedInstance) {
        _sharedInstance = [[SRSoundManager alloc] init];
    }
    
    return _sharedInstance;
}

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Setup sound
    NSArray*        actionNames;
    NSEnumerator*   enumerator;
    NSString*       actionName;
    actionNames = [NSArray arrayWithObjects:
            SRUniversalAccessSoundPageLoadDone, SRUniversalAccessSoundPageLoadError, 
            SRUniversalAccessSoundJavaScriptDialog, 
            nil];
    enumerator = [actionNames objectEnumerator];
    while (actionName = [enumerator nextObject]) {
        // Set sound
        NSString*   soundPath;
        soundPath = [defaults stringForKey:actionName];
        if (soundPath) {
            [self setSound:soundPath forAction:actionName];
        }
    }
    
    return self;
}

//--------------------------------------------------------------//
#pragma mark -- Sound --
//--------------------------------------------------------------//

- (SystemSoundActionID)soundForAction:(NSString*)actionName
{
    // Return action ID
    if ([actionName isEqualToString:SRUniversalAccessSoundPageLoadDone]) {
        return _pageLoadDoneSoundActionId;
    }
    if ([actionName isEqualToString:SRUniversalAccessSoundPageLoadError]) {
        return _pageLoadErrorSoundActionId;
    }
    if ([actionName isEqualToString:SRUniversalAccessSoundJavaScriptDialog]) {
        return _javaScriptDialogSoundActionId;
    }
    
    return 0;
}

- (SystemSoundActionID)setSound:(NSString*)soundFilePath forAction:(NSString*)actionName
{
    if (!soundFilePath || !actionName) {
        return 0;
    }
    
    // Get action ID variable
    SystemSoundActionID*    actionId = NULL;
    if ([actionName isEqualToString:SRUniversalAccessSoundPageLoadDone]) {
        actionId = &_pageLoadDoneSoundActionId;
    }
    if ([actionName isEqualToString:SRUniversalAccessSoundPageLoadError]) {
        actionId = &_pageLoadErrorSoundActionId;
    }
    if ([actionName isEqualToString:SRUniversalAccessSoundJavaScriptDialog]) {
        actionId = &_javaScriptDialogSoundActionId;
    }
    
    if (!actionId) {
        return 0;
    }
    
    SystemSoundActionID soundActionId = 0;
    
    // For 'None'
    if ([soundFilePath isEqualToString:@"None"]) {
        if (*actionId) {
            SystemSoundRemoveActionID(*actionId);
        }
        *actionId = 0;
    }
    else {
        // Create action ID
        OSStatus            err;
        FSRef               soundFileRef;
        err = FSPathMakeRef((const UInt8*)[soundFilePath fileSystemRepresentation], &soundFileRef, NULL);
        if (err != noErr) {
            return 0;
        }
        err = SystemSoundGetActionID(&soundFileRef, &soundActionId);
        if (err != noErr) {
            return 0;
        }
        
        // Set action ID
        if (*actionId) {
            SystemSoundRemoveActionID(*actionId);
        }
        *actionId = soundActionId;
    }
    
    // Update defaults
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    if (![soundFilePath isEqualToString:[defaults stringForKey:actionName]]) {
        [defaults setObject:soundFilePath forKey:actionName];
    }
    
    return soundActionId;
}

- (void)playSoundForAction:(NSString*)actionName
{
    // Get sound action ID
    SystemSoundActionID soundActionId;
    soundActionId = [self soundForAction:actionName];
    if (!soundActionId) {
        return;
    }
    
    // Sound test
    SystemSoundPlay(soundActionId);
}

@end
